#!/usr/bin/env python3

"""Functions to compute the model metrics."""

from __future__ import print_function, division
import sys
import math

from rawdata import *
from collections import OrderedDict

# error import variables
error_import_pandas = False
error_import_seaborn = False
error_import_matplotlib = False
error_import_numpy = False

try:
    import numpy as np
except ImportError:
    error_import_numpy = True


try:
    import pandas as pd
except ImportError:
    error_import_pandas = True

try:
    import seaborn as sns
except ImportError:
    error_import_seaborn = True


try:
    import matplotlib.pyplot as plt
except ImportError:
    error_import_matplotlib = True
    


# Contains all model factor entries with a printable name.
# This is used to generate and print all model factors, so, if an entry is added,
# it should be added here, too.

other_metrics_doc = OrderedDict([('elapsed_time', 'Elapsed time (sec)'),
                               ('efficiency', 'Efficiency'),
                               ('speedup', 'Speedup'),
                               ('ipc', 'Average IPC'),
                               ('freq', 'Average frequency (GHz)'),
                               ('flushing', 'Flushing (%)'),
                               ('io_mpiio', 'MPI I/O (%)'),
                               ('io_posix', 'POSIX I/O (%)'),
                               ('io_eff', 'I/O Efficiency (%)')])

mod_factors_doc = OrderedDict([('global_eff', 'Global efficiency'),
                               ('parallel_eff', '-- Parallel efficiency'),
                               ('load_balance', '   -- Load balance'),
                               ('comm_eff', '   -- Communication efficiency'),
                               ('serial_eff', '      -- Serialization efficiency'),
                               ('transfer_eff', '      -- Transfer efficiency'),
                               ('comp_scale', '-- Computation scalability'),
                               ('ipc_scale', '   -- IPC scalability'),
                               ('inst_scale', '   -- Instruction scalability'),
                               ('freq_scale', '   -- Frequency scalability')])

mod_factors_scale_plus_io_doc = OrderedDict([('comp_scale', '-- Computation scalability + I/O'),
                               ('ipc_scale', '   -- IPC scalability'),
                               ('inst_scale', '   -- Instruction scalability'),
                               ('freq_scale', '   -- Frequency scalability')])

mod_hybrid_factors_doc = OrderedDict([
                               ('hybrid_eff', '-- Hybrid Parallel efficiency'),
                               ('mpi_parallel_eff', '   -- MPI Parallel efficiency'),
                               ('mpi_load_balance', '      -- MPI Load balance'),
                               ('mpi_comm_eff', '      -- MPI Communication efficiency'),
                               ('serial_eff', '         -- Serialization efficiency'),
                               ('transfer_eff', '         -- Transfer efficiency'),
                               ('omp_parallel_eff', '   -- OpenMP Parallel efficiency'),
                               ('omp_load_balance', '      -- OpenMP Load balance'),
                               ('omp_comm_eff', '      -- OpenMP Communication efficiency')])

mod_device_factors_doc = OrderedDict([
                               ('dev_global_eff', 'DEVICE Global efficiency'),
                               ('dev_parallel_eff', '-- Device Parallel efficiency'),
                               ('dev_load_balance', '   == Device Load balance'),
                               ('dev_comm_eff', '   == Device Communication efficiency'),
                               ('dev_orches_eff', '   == Device Orchestration efficiency'),
                               ('dev_comp_scale', '-- Device Computation scalability')])

mod_host_factors_doc = OrderedDict([
                               ('host_global_eff', 'HOST Global efficiency'),
                               ('host_parallel_eff', '-- Host Parallel efficiency'),
                               ('mpi_parallel_eff', '   == MPI Parallel efficiency'),
                               ('mpi_load_balance', '       -- MPI Load balance'),
                               ('mpi_comm_eff', '       -- MPI Communication efficiency'),
                               ('serial_eff', '          -- Serialization efficiency'),
                               ('transfer_eff', '          -- Transfer efficiency'),
                               ('dev_offload_eff', '   == Device Offload efficiency'),
                               ('host_comp_scale', '-- Host Computation scalability'),
                               ('host_ipc_scale', '   == IPC scalability'),
                               ('host_inst_scale', '   == Instruction scalability'),
                               ('host_freq_scale', '   == Frequency scalability')])

def create_mod_factors(trace_list):
    """Creates 2D dictionary of the model factors and initializes with an empty
    string. The mod_factors dictionary has the format: [mod factor key][trace].
    """
    global mod_factors_doc
    mod_factors = {}
    for key in mod_factors_doc:
        trace_dict = {}
        for trace_name in trace_list:
            trace_dict[trace_name] = 0.0
        mod_factors[key] = trace_dict

    return mod_factors


def create_mod_factors_scale_io(trace_list):
    """Creates 2D dictionary of the model factors and initializes with an empty
    string. The mod_factors dictionary has the format: [mod factor key][trace].
    """
    global mod_factors_scale_plus_io_doc
    mod_factors_scale_plus_io = {}
    for key in mod_factors_scale_plus_io_doc:
        trace_dict = {}
        for trace_name in trace_list:
            trace_dict[trace_name] = 0.0
        mod_factors_scale_plus_io[key] = trace_dict

    return mod_factors_scale_plus_io


def create_hybrid_mod_factors(trace_list):
    """Creates 2D dictionary of the hybrid model factors and initializes with an empty
    string. The hybrid_factors dictionary has the format: [mod factor key][trace].
    """
    global mod_hybrid_factors_doc
    hybrid_factors = {}
    for key in mod_hybrid_factors_doc:
        trace_dict = {}
        for trace_name in trace_list:
            trace_dict[trace_name] = 0.0
        hybrid_factors[key] = trace_dict

    return hybrid_factors

def create_mod_device_factors(trace_list):
    """Creates 2D dictionary of the hybrid model factors at device level and initializes with an empty
    string. The hybrid_factors dictionary has the format: [mod factor key][trace].
    """
    global mod_device_factors_doc
    device_factors = {}
    for key in mod_device_factors_doc:
        trace_dict = {}
        for trace_name in trace_list:
            trace_dict[trace_name] = 0.0
        device_factors[key] = trace_dict

    return device_factors

def create_mod_host_factors(trace_list):
    """Creates 2D dictionary of the hybrid model factors at device level and initializes with an empty
    string. The hybrid_factors dictionary has the format: [mod factor key][trace].
    """
    global mod_host_factors_doc
    host_factors = {}
    for key in mod_host_factors_doc:
        trace_dict = {}
        for trace_name in trace_list:
            trace_dict[trace_name] = 0.0
        host_factors[key] = trace_dict

    return host_factors


def create_other_metrics(trace_list):
    """Creates 2D dictionary of the other metrics and initializes with an empty
    string. The other_metrics dictionary has the format: [mod factor key][trace].
    """
    global other_metrics_doc
    other_metrics = {}
    for key in other_metrics_doc:
        trace_dict = {}
        for trace_name in trace_list:
            trace_dict[trace_name] = 0.0
        other_metrics[key] = trace_dict

    return other_metrics


def get_scaling_type(raw_data, trace_list, trace_processes, cmdl_args):
    """Guess the scaling type (weak/strong) based on the useful instructions.
    Computes the normalized instruction ratio for all measurements, whereas the
    normalized instruction ratio is (instructions ratio / process ratio) with
    the smallest run as reference. For exact weak scaling the normalized ratio
    should be exactly 1 and for exact strong scaling it should be close to zero
    with an upper bound of 0.5. The eps value defines the threshold to be
    considered weak scaling and should give enough buffer to safely handle
    non-ideal scaling.
    """
    eps = 0.9
    normalized_inst_ratio = 0
    normalized_runtime_ratio = 0
    normalized_useful_avg_ratio = 0

    # Check if there is only one trace.
    if len(trace_list) == 1:
        return 'strong'

    for trace in trace_list:
        try:  # except NaN
            inst_ratio = float(raw_data['useful_ins'][trace]) / float(raw_data['useful_ins'][trace_list[0]])
        except:
            inst_ratio = 0.0
        try:  # except NaN
            proc_ratio = float(trace_processes[trace]) / float(trace_processes[trace_list[0]])
        except:
            proc_ratio = 'NaN'
        try:  # except NaN
            runtime_ratio = float(raw_data['runtime'][trace]) / float(raw_data['runtime'][trace_list[0]])
        except:
            runtime_ratio = 'NaN'
        try:  # except NaN
            useful_avg_ratio = float(raw_data['useful_avg'][trace]) / float(raw_data['useful_avg'][trace_list[0]])
        except:
            useful_avg_ratio = 'NaN'

        normalized_inst_ratio += inst_ratio / proc_ratio
        normalized_runtime_ratio += runtime_ratio
        normalized_useful_avg_ratio += useful_avg_ratio
    # Get the average inst increase. Ignore ratio of first trace 1.0)
    normalized_inst_ratio = (normalized_inst_ratio - 1) / (len(trace_list) - 1)
    normalized_runtime_ratio = (normalized_runtime_ratio - 1) / (len(trace_list) - 1)
    normalized_useful_avg_ratio = (normalized_useful_avg_ratio - 1) / (len(trace_list) - 1)

    scaling_computed = ''
    scale_type = 0

    if normalized_inst_ratio > eps:
        scale_type += 1
    if normalized_runtime_ratio > eps:
        scale_type += 1
    if normalized_useful_avg_ratio > eps:
        scale_type += 1

    # print("normalized_inst_ratio: ", normalized_inst_ratio)
    # print("normalized_runtime_ratio: ", normalized_runtime_ratio)
    # print("normalized_useful_avg_ratio: ", normalized_useful_avg_ratio)

    # scale_type greater than 1 means weak scaling.
    if scale_type > 1:
        scaling_computed = 'weak'
    else:
        scaling_computed = 'strong'

    if cmdl_args.scaling == 'auto':
        if cmdl_args.debug:
            print('==DEBUG== Detected ' + scaling_computed + ' scaling.')
            print('')
        return scaling_computed

    if cmdl_args.scaling == 'weak':
        if scaling_computed == 'strong':
            print('==Warning== Scaling set to weak scaling but detected strong scaling.')
            print('')
        return 'weak'

    if cmdl_args.scaling == 'strong':
        if scaling_computed == 'weak':
            print('==Warning== Scaling set to strong scaling but detected weak scaling.')
            print('')
        return 'strong'

    print('==Error== reached undefined control flow state.')
    sys.exit(1)


def compute_model_factors(raw_data, trace_list, trace_processes, trace_mode, list_mpi_procs_count, cmdl_args):
    """Computes the model factors from the gathered raw data and returns the
    according dictionary of model factors."""
    mod_factors = create_mod_factors(trace_list)
    hybrid_factors = create_hybrid_mod_factors(trace_list)
    other_metrics = create_other_metrics(trace_list)
    mod_factors_scale_plus_io = create_mod_factors_scale_io(trace_list)
    device_factors = create_mod_device_factors(trace_list)
    host_factors = create_mod_host_factors(trace_list)
    # Guess the weak or strong scaling
    scaling = get_scaling_type(raw_data, trace_list, trace_processes, cmdl_args)

    # Loop over all traces
    for trace in trace_list:

        proc_ratio = float(trace_processes[trace]) / float(trace_processes[trace_list[0]])
        total_procs = trace_processes[trace]

        # Flushing measurements
        try:  # except NaN
            if trace_mode[trace][0:len("Burst")] != 'Burst':
                other_metrics['flushing'][trace] = float(raw_data['flushing_tot'][trace]) \
                                                     / (raw_data['runtime'][trace] * total_procs) * 100.0
            else:
                other_metrics['flushing'][trace] = 0.0
        except:
            other_metrics['flushing'][trace] = 0.0

        # I/O measurements
        try:  # except NaN
            if trace_mode[trace][0:len("Burst")] != 'Burst':
                other_metrics['io_mpiio'][trace] = float(raw_data['mpiio_tot'][trace]) \
                                               / (raw_data['runtime'][trace] * total_procs) * 100.0
            else:
                other_metrics['io_mpiio'][trace] = 0.0
        except:
            other_metrics['io_mpiio'][trace] = 0.0

        try:  # except NaN
            if trace_mode[trace][0:len("Burst")] != 'Burst':
                other_metrics['io_posix'][trace] = float(raw_data['io_tot'][trace]) \
                                               / (raw_data['runtime'][trace] * total_procs) * 100.0
            else:
                other_metrics['io_posix'][trace] = 0.0
        except:
            other_metrics['io_posix'][trace] = 0.0
        try:  # except NaN

            io_total = float(raw_data['mpiio_tot'][trace] + raw_data['flushing_tot'][trace]
                             + raw_data['io_tot'][trace])
            other_metrics['io_eff'][trace] = float(raw_data['useful_tot'][trace]) \
                                             / (raw_data['useful_tot'][trace] + io_total) * 100.0
        except:
            other_metrics['io_eff'][trace] = 0.0

        # Basic efficiency factors
        try:  # except NaN
            if other_metrics['io_posix'][trace] > 0.0 or other_metrics['flushing'][trace] > 5.0:
                mod_factors['load_balance'][trace] = float(raw_data['useful_plus_io_avg'][trace]) \
                                                     / float(raw_data['useful_plus_io_max'][trace]) * 100.0
            else:
                mod_factors['load_balance'][trace] = float(raw_data['useful_avg'][trace]) \
                                                 / float(raw_data['useful_max'][trace]) * 100.0
        except:
            mod_factors['load_balance'][trace] = 'NaN'

        try:  # except NaN
            if other_metrics['io_posix'][trace] > 0.0 or other_metrics['flushing'][trace] > 5.0:
                mod_factors['comm_eff'][trace] = float(raw_data['useful_plus_io_max'][trace] \
                                                 / raw_data['runtime'][trace] * 100.0)
            else:
                mod_factors['comm_eff'][trace] = float(raw_data['useful_max'][trace] \
                                                     / raw_data['runtime'][trace] * 100.0)
        except:
            mod_factors['comm_eff'][trace] = 'NaN'

        try:  # except NaN
            if trace_mode[trace][0:len("Burst")] != 'Burst':
                if other_metrics['io_posix'][trace] > 0.0 or other_metrics['flushing'][trace] > 5.0:
                    mod_factors['parallel_eff'][trace] = float(raw_data['useful_plus_io_avg'][trace]) \
                                                     / float(raw_data['runtime'][trace]) * 100.0
                else:
                    mod_factors['parallel_eff'][trace] = float(mod_factors['load_balance'][trace]) \
                                                     * float(mod_factors['comm_eff'][trace]) / 100.0
            elif trace_mode[trace] == 'Burst+MPI':
                mod_factors['parallel_eff'][trace] = float(raw_data['burst_useful_avg'][trace]) \
                                                     / float(raw_data['runtime'][trace]) * 100.0
        except:
            mod_factors['parallel_eff'][trace] = 'NaN'

        try:  # except NaN
            if len(trace_list) > 1:
                if trace_mode[trace][0:len("Burst")] == 'Burst':
                    if trace_mode[trace_list[0]][0:len("Burst")] == 'Burst':
                        if scaling == 'strong':
                            mod_factors['comp_scale'][trace] = raw_data['burst_useful_tot'][trace_list[0]] \
                                                   / raw_data['burst_useful_tot'][trace] * 100.0
                        else:
                            mod_factors['comp_scale'][trace] = raw_data['burst_useful_tot'][trace_list[0]] \
                                                   / raw_data['burst_useful_tot'][trace] \
                                                   * proc_ratio * 100.0
                    else:
                        mod_factors['comp_scale'][trace] = 100.0
                else:
                    if scaling == 'strong':
                        mod_factors['comp_scale'][trace] = float(raw_data['useful_tot'][trace_list[0]]) \
                                                   / float(raw_data['useful_tot'][trace]) * 100.0
                    else:
                        mod_factors['comp_scale'][trace] = float(raw_data['useful_tot'][trace_list[0]]) \
                                                   / float(raw_data['useful_tot'][trace]) * proc_ratio * 100.0
            else:
                mod_factors['comp_scale'][trace] = 'Non-Avail'
        except:
            mod_factors['comp_scale'][trace] = 'NaN'

        # Computation Scale + Serial I/O
        try:  # except NaN
            if len(trace_list) > 1:
                if other_metrics['io_posix'][trace] > 0.0 or other_metrics['flushing'][trace] > 0.0:
                    io_serial_0 = float(raw_data['io_tot'][trace_list[0]] + raw_data['flushing_tot'][trace_list[0]])
                    io_serial_n = float(raw_data['io_tot'][trace] + raw_data['flushing_tot'][trace])
                    if scaling == 'strong':
                        mod_factors_scale_plus_io['comp_scale'][trace] = float((raw_data['useful_tot'][trace_list[0]]
                                                                      + io_serial_0) / (raw_data['useful_tot'][trace]
                                                                                        + io_serial_n) * 100.0)
                    else:
                        mod_factors_scale_plus_io['comp_scale'][trace] = (float(raw_data['useful_tot'][trace_list[0]])
                                                                          + io_serial_0) \
                                                                     / (float(raw_data['useful_tot'][trace])
                                                                        + io_serial_n) * proc_ratio * 100.0
                else:
                    mod_factors_scale_plus_io['comp_scale'][trace] = float(mod_factors['comp_scale'][trace])
            else:
                mod_factors_scale_plus_io['comp_scale'][trace] = 'Non-Avail'
        except:
            mod_factors_scale_plus_io['comp_scale'][trace] = 'NaN'


        try:  # except NaN
            if len(trace_list) > 1:
                mod_factors['global_eff'][trace] = float(mod_factors['parallel_eff'][trace]) \
                                                   * float(mod_factors['comp_scale'][trace]) / 100.0
            else:
                mod_factors['global_eff'][trace] = float(mod_factors['parallel_eff'][trace]) * 100.0 / 100.0
        except:
            mod_factors['global_eff'][trace] = 'NaN'

        # Hybrid metrics calculation
        # Control OutMPI values no availables
        if math.isnan(float(raw_data['outsidempi_avg'][trace])) or math.isnan(float(raw_data['outsidempi_max'][trace])):
            outmpi_measures = False
        else:
            outmpi_measures = True

            # ------->  MPI metrics
        try:  # except NaN
            if not outmpi_measures:
                hybrid_factors['mpi_load_balance'][trace] = 'Non-Avail'
            elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                hybrid_factors['mpi_load_balance'][trace] = float(raw_data['outsidempi_avg'][trace]) \
                                                        / float(raw_data['outsidempi_max'][trace]) * 100.0
            else:
                hybrid_factors['mpi_load_balance'][trace] = 'N/A'
        except:
                hybrid_factors['mpi_load_balance'][trace] = 'NaN'

        try:  # except NaN
            if not outmpi_measures:
                hybrid_factors['mpi_comm_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                hybrid_factors['mpi_comm_eff'][trace] = float(raw_data['outsidempi_max'][trace]) \
                                                    / float(raw_data['runtime'][trace]) * 100.0
            else:
                hybrid_factors['mpi_comm_eff'][trace] = 'N/A'
        except:
            hybrid_factors['mpi_comm_eff'][trace] = 'NaN'

        # ------------> BEGIN MPI communication sub-metrics
        try:  # except NaN
            if (not outmpi_measures) or cmdl_args.skip_simulation:
                hybrid_factors['serial_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace] == "Detailed+MPI+OpenMP" \
                        or trace_mode[trace] == 'Detailed+MPI+CUDA':
                hybrid_factors['serial_eff'][trace] = float(raw_data['outsidempi_dim'][trace]) \
                                               / float(raw_data['runtime_dim'][trace]) * 100.0
                mod_factors['serial_eff'][trace] = 'N/A'
            elif trace_mode[trace] == "Detailed+MPI":
                mod_factors['serial_eff'][trace] = float(raw_data['useful_dim'][trace]) \
                                                   / float(raw_data['runtime_dim'][trace]) * 100.0
                hybrid_factors['serial_eff'][trace] = 'N/A'
            elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                mod_factors['serial_eff'][trace] = 'N/A'
                hybrid_factors['serial_eff'][trace] = 'Non-Avail'
            else:
                mod_factors['serial_eff'][trace] = 'Non-Avail'
                hybrid_factors['serial_eff'][trace] = 'N/A'

            if round(hybrid_factors['serial_eff'][trace]) > 100:
                hybrid_factors['serial_eff'][trace] = 'Warning!'
        except:    
            if trace_mode[trace] == 'Detailed+MPI' or trace_mode[trace] == 'Detailed+MPI+OpenMP' \
                    or trace_mode[trace] == 'Detailed+MPI+CUDA':
                if hybrid_factors['serial_eff'][trace] != 'Non-Avail' and hybrid_factors['serial_eff'][trace] != 'N/A':
                    hybrid_factors['serial_eff'][trace] = 'NaN'

        try:  # except NaN
            if (not outmpi_measures) or cmdl_args.skip_simulation:
                hybrid_factors['transfer_eff'][trace] = 'Non-Avail'
            elif hybrid_factors['serial_eff'][trace] != 'Warning!':
                if trace_mode[trace] == "Detailed+MPI+OpenMP" \
                        or trace_mode[trace] == 'Detailed+MPI+CUDA':
                    hybrid_factors['transfer_eff'][trace] = float(hybrid_factors['mpi_comm_eff'][trace]) \
                                                        / float(hybrid_factors['serial_eff'][trace]) * 100.0
                    mod_factors['transfer_eff'][trace] = 'N/A'
                elif trace_mode[trace] == "Detailed+MPI":
                    mod_factors['transfer_eff'][trace] = float(mod_factors['comm_eff'][trace]) \
                                                         / float(mod_factors['serial_eff'][trace]) * 100.0
                    hybrid_factors['transfer_eff'][trace] = 'N/A'
                elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                    hybrid_factors['transfer_eff'][trace] = 'Non-Avail'
                    mod_factors['transfer_eff'][trace] = 'N/A'
                else:
                    hybrid_factors['transfer_eff'][trace] = 'N/A'
                    mod_factors['transfer_eff'][trace] = 'Non-Avail'

                if not hybrid_factors['transfer_eff'][trace] == 'Non-Avail':
                    if round(hybrid_factors['transfer_eff'][trace]) > 100:
                        hybrid_factors['transfer_eff'][trace] = 'Warning!'
        except:
            if trace_mode[trace] == 'Detailed+MPI' or trace_mode[trace] == 'Detailed+MPI+OpenMP' \
                    or trace_mode[trace] == 'Detailed+MPI+CUDA':
                if hybrid_factors['transfer_eff'][trace] != 'N/A' and hybrid_factors['transfer_eff'][trace] != 'Non-Avail':
                    hybrid_factors['transfer_eff'][trace] = 'NaN'

        # --------------> END MPI communication sub-metrics
        try:  # except NaN
            if not outmpi_measures:
                hybrid_factors['mpi_parallel_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                if raw_data['outsidempi_tot'][trace] == raw_data['outsidempi_tot_diff'][trace]:
                    hybrid_factors['mpi_parallel_eff'][trace] = float(raw_data['outsidempi_tot'][trace] /
                                                                  (raw_data['runtime'][trace]
                                                                   * list_mpi_procs_count[trace]) * 100.0)
                else:
                    hybrid_factors['mpi_parallel_eff'][trace] = float(raw_data['outsidempi_tot_diff'][trace]
                                                                  / (raw_data['runtime'][trace]
                                                                     * trace_processes[trace]) * 100.0)
            else:
                hybrid_factors['mpi_parallel_eff'][trace] = 'N/A'
        except:
            hybrid_factors['mpi_parallel_eff'][trace] = 'NaN'

        # ------->  Metrics for the second parallel paradigm
        try:  # except NaN
            if not outmpi_measures:
                hybrid_factors['omp_comm_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                hybrid_factors['omp_comm_eff'][trace] = float(mod_factors['comm_eff'][trace]) \
                                                        / float(hybrid_factors['mpi_comm_eff'][trace]) * 100.0
            else:
                hybrid_factors['omp_comm_eff'][trace] = 'N/A'
        except:
            hybrid_factors['omp_comm_eff'][trace] = 'NaN'

        try:  # except NaN
            if not outmpi_measures:
                hybrid_factors['omp_load_balance'][trace] = 'Non-Avail'
            elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                hybrid_factors['omp_load_balance'][trace] = float(mod_factors['load_balance'][trace]) \
                                                            / float(hybrid_factors['mpi_load_balance'][trace]) * 100.0
            else:
                hybrid_factors['omp_load_balance'][trace] = 'N/A'
        except:
            hybrid_factors['omp_load_balance'][trace] = 'NaN'

        try:  # except NaN
            if not outmpi_measures:
                hybrid_factors['omp_parallel_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                hybrid_factors['omp_parallel_eff'][trace] = float(mod_factors['parallel_eff'][trace]) \
                                                            / float(hybrid_factors['mpi_parallel_eff'][trace]) * 100.0
            else:
                hybrid_factors['omp_parallel_eff'][trace] = 'N/A'
        except:
            hybrid_factors['omp_parallel_eff'][trace] = 'NaN'

# ------->  HOST metrics
        try:  # except NaN
            if outmpi_measures and (trace_mode[trace] == 'Detailed+MPI+CUDA'):
                host_factors['mpi_parallel_eff'][trace] = hybrid_factors['mpi_parallel_eff'][trace]
                host_factors['mpi_comm_eff'][trace] = hybrid_factors['mpi_comm_eff'][trace]
                host_factors['mpi_load_balance'][trace] = hybrid_factors['mpi_load_balance'][trace]
            else:
                host_factors['mpi_parallel_eff'][trace] = 'Non-Avail'
                host_factors['mpi_comm_eff'][trace] = 'Non-Avail'
                host_factors['mpi_load_balance'][trace] = 'Non-Avail'
        except:
            host_factors['mpi_parallel_eff'][trace] = 'NaN'
            host_factors['mpi_comm_eff'][trace] = 'NaN'
            host_factors['mpi_load_balance'][trace] = 'NaN'

# ------------> BEGIN MPI communication sub-metrics HOST
        try:  # except NaN
            if not outmpi_measures or cmdl_args.skip_simulation:
                host_factors['serial_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace] == 'Detailed+MPI+CUDA':
                host_factors['serial_eff'][trace] = hybrid_factors['serial_eff'][trace]
        except:
            host_factors['serial_eff'][trace] = 'NaN'        

        try:  # except NaN
            if not outmpi_measures or cmdl_args.skip_simulation:
                host_factors['transfer_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace] == 'Detailed+MPI+CUDA':
                    host_factors['transfer_eff'][trace] = hybrid_factors['transfer_eff'][trace]
        except:
            host_factors['transfer_eff'][trace] = 'NaN'

        # --------------> END MPI communication sub-metrics
        
        ### Offloading
        try:  # except NaN
            if not outmpi_measures:
                host_factors['dev_offload_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace] == 'Detailed+MPI+CUDA':
                host_factors['dev_offload_eff'][trace] = 100 * (float(raw_data['useful_host'][trace])\
                /float(raw_data['outsidempi_tot'][trace]) )
            else:
                host_factors['dev_offload_eff'][trace] = 'N/A'
        except:
            host_factors['dev_offload_eff'][trace] = 'NaN'
        
        ### Host Parallel Efficiency
        try:  # except NaN
            if not outmpi_measures:
                host_factors['host_parallel_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace] == 'Detailed+MPI+CUDA':
                host_factors['host_parallel_eff'][trace] = (host_factors['mpi_parallel_eff'][trace]/100) \
                * (host_factors['dev_offload_eff'][trace]/100) * 100
            else:
                host_factors['host_parallel_eff'][trace] = 'N/A'
        except:
            host_factors['host_parallel_eff'][trace] = 'NaN'    

# ------->  Devices metrics
        try:  # except NaN
            if not outmpi_measures:
                device_factors['dev_parallel_eff'][trace] = 'Non-Avail'
                device_factors['dev_load_balance'][trace] = 'Non-Avail'
                device_factors['dev_comm_eff'][trace] = 'Non-Avail'
                device_factors['dev_orches_eff'][trace] = 'Non-Avail'
                device_factors['dev_offload_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace] == 'Detailed+MPI+CUDA':
                device_factors['dev_parallel_eff'][trace] = 100 * (float(raw_data['useful_device'][trace])\
                /(int(raw_data['count_devices'][trace])*float(raw_data['runtime'][trace])))
                device_factors['dev_load_balance'][trace] = 100 * ( (float(raw_data['useful_device'][trace])/int(raw_data['count_devices'][trace]))\
                /float(raw_data['useful_device_max'][trace]) )
                device_factors['dev_comm_eff'][trace] = 100 * ( float(raw_data['useful_device_max'][trace])\
                /float(raw_data['useful_memtransf_device_max'][trace]) )
                device_factors['dev_orches_eff'][trace] = 100 * ( float(raw_data['useful_memtransf_device_max'][trace]) \
                / float(raw_data['runtime'][trace]) )
            else:
                device_factors['dev_parallel_eff'][trace] = 'N/A'
                device_factors['dev_load_balance'][trace] = 'N/A'
                device_factors['dev_comm_eff'][trace] = 'N/A'
                device_factors['dev_orches_eff'][trace] = 'N/A'   
        except:
            device_factors['dev_parallel_eff'][trace] = 'NaN'
            device_factors['dev_load_balance'][trace] = 'NaN'
            device_factors['dev_comm_eff'][trace] = 'NaN'
            device_factors['dev_orches_eff'][trace] = 'NaN'           
        
        # Device Computation Scalability
        try:  # except NaN
            if outmpi_measures and (trace_mode[trace] == 'Detailed+MPI+CUDA') and (len(trace_list) > 1):
                if scaling == 'strong':
                        device_factors['dev_comp_scale'][trace] = float(raw_data['useful_device'][trace_list[0]]) \
                                                   / float(raw_data['useful_device'][trace]) * 100.0
                else:
                    device_factors['dev_comp_scale'][trace] = float(raw_data['useful_device'][trace_list[0]]) \
                                                   / float(raw_data['useful_device'][trace]) * proc_ratio * 100.0
            else:
                device_factors['dev_comp_scale'][trace] = 'Non-Avail'
        except:
               device_factors['dev_comp_scale'][trace] = 'NaN'
        
        # Device Global Efficiency
        try:  # except NaN
            if outmpi_measures and (trace_mode[trace] == 'Detailed+MPI+CUDA'):
                if (len(trace_list) > 1):
                    device_factors['dev_global_eff'][trace] = (device_factors['dev_parallel_eff'][trace]/100) \
                    * (device_factors['dev_comp_scale'][trace]/100) * 100
                else:
                    device_factors['dev_global_eff'][trace] = device_factors['dev_parallel_eff'][trace]
        except:
            device_factors['dev_global_eff'][trace] = 'NaN'

# ------->  Global Hybrid Metric
        try:  # except NaN
            if not outmpi_measures:
                hybrid_factors['hybrid_eff'][trace] = 'Non-Avail'
            elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                hybrid_factors['hybrid_eff'][trace] = float(hybrid_factors['mpi_parallel_eff'][trace]) \
                                               * float(hybrid_factors['omp_parallel_eff'][trace]) / 100.0
            else:
                hybrid_factors['hybrid_eff'][trace] = 'N/A'
        except:
            hybrid_factors['hybrid_eff'][trace] = 'NaN'

        # Basic scalability factors - Computational
        # IPC Scalability
        try:  # except NaN
            other_metrics['ipc'][trace] = float(raw_data['useful_ins'][trace]) \
                                        / float(raw_data['useful_cyc'][trace])
        except:
            if (raw_data['useful_ins'][trace] == 0) or (raw_data['useful_cyc'][trace] == 0):
                other_metrics['ipc'][trace] = 'Non-Avail'
            else:
                other_metrics['ipc'][trace] = 'NaN'
        try:  # except NaN
            if len(trace_list) > 1:
                if trace_mode[trace][:5] != 'Burst' and trace_mode[trace] != 'Sampling':
                    mod_factors['ipc_scale'][trace] = float(other_metrics['ipc'][trace]) \
                                              / float(other_metrics['ipc'][trace_list[0]]) * 100.0
                else:
                    mod_factors['ipc_scale'][trace] = 'Non-Avail'
            else:
                mod_factors['ipc_scale'][trace] = 'Non-Avail'
        except:
            mod_factors['ipc_scale'][trace] = 'NaN'

        # IPC scale + Serial I/O
        try:  # except NaN
            if len(trace_list) > 1:
                if other_metrics['io_posix'][trace] > 0.0 or other_metrics['flushing'][trace] > 0.0:
                    ipc_serial_io_0 = (raw_data['useful_ins'][trace_list[0]] + raw_data['io_ins'][trace_list[0]]
                                       + raw_data['flushing_ins'][trace_list[0]]) \
                                      / (raw_data['useful_cyc'][trace_list[0]] + raw_data['io_cyc'][trace_list[0]]
                                         + raw_data['flushing_cyc'][trace_list[0]])

                    ipc_serial_io_n = (raw_data['useful_ins'][trace] + raw_data['io_ins'][trace]
                                       + raw_data['flushing_ins'][trace])\
                                      / (raw_data['useful_cyc'][trace] + raw_data['io_cyc'][trace]
                                         + raw_data['flushing_cyc'][trace])
                    mod_factors_scale_plus_io['ipc_scale'][trace] = float(ipc_serial_io_n / ipc_serial_io_0 * 100.0)
                else:
                    mod_factors_scale_plus_io['ipc_scale'][trace] = float(mod_factors['ipc_scale'][trace])
            else:
                mod_factors_scale_plus_io['ipc_scale'][trace] = 'Non-Avail'
        except:
            mod_factors_scale_plus_io['ipc_scale'][trace] = 'NaN'

        # Frequency Scalability
        try:  # except NaN
            if (raw_data['useful_cyc'][trace] == 0):
                other_metrics['freq'][trace] = 'Non-Avail'
            else:
                other_metrics['freq'][trace] = float(raw_data['useful_cyc'][trace]) \
                                           / float(raw_data['useful_not_0_tot'][trace]) / 1000
        except:
            other_metrics['freq'][trace] = 'NaN'
        try:  # except NaN
            if len(trace_list) > 1:
                if trace_mode[trace][:5] != 'Burst' and trace_mode[trace] != 'Sampling':
                    mod_factors['freq_scale'][trace] = float(other_metrics['freq'][trace]) \
                                               / float(other_metrics['freq'][trace_list[0]]) * 100.0
                else:
                    mod_factors['freq_scale'][trace] = 'Non-Avail'
            else:
                mod_factors['freq_scale'][trace] = 'Non-Avail'
        except:
            mod_factors['freq_scale'][trace] = 'NaN'

        # freq scale + Serial I/O
        try:  # except NaN
            if len(trace_list) > 1:
                if other_metrics['io_posix'][trace] > 0.0 or other_metrics['flushing'][trace] > 0.0:
                    freq_serial_io_0 = (float(raw_data['useful_cyc'][trace_list[0]])
                                        + float(raw_data['io_cyc'][trace_list[0]])
                                        + float(raw_data['flushing_cyc'][trace_list[0]]))\
                                       / (raw_data['useful_not_0_tot'][trace_list[0]]
                                          + float(raw_data['io_tot'][trace_list[0]])
                                          + float(raw_data['flushing_tot'][trace_list[0]])) / 1000

                    freq_serial_io_n = (float(raw_data['useful_cyc'][trace])
                                              + float(raw_data['io_cyc'][trace])
                                              + float(raw_data['flushing_cyc'][trace]))\
                                       / (float(raw_data['useful_not_0_tot'][trace])
                                          + float(raw_data['io_tot'][trace])
                                          + float(raw_data['flushing_tot'][trace])) / 1000
                    mod_factors_scale_plus_io['freq_scale'][trace] = float(freq_serial_io_n / freq_serial_io_0 * 100.0)
                else:
                    mod_factors_scale_plus_io['freq_scale'][trace] = float(mod_factors['freq_scale'][trace])
            else:
                mod_factors_scale_plus_io['freq_scale'][trace] = 'Non-Avail'
        except:
            mod_factors_scale_plus_io['freq_scale'][trace] = 'NaN'

    # Instruction Scalability
        try:  # except NaN
            if len(trace_list) > 1:
                if trace_mode[trace][:5] != 'Burst' and trace_mode[trace] != 'Sampling':
                    if scaling == 'strong':
                        mod_factors['inst_scale'][trace] = float(raw_data['useful_ins'][trace_list[0]])\
                                                       / float(raw_data['useful_ins'][trace]) * 100.0
                    else:
                        procs_ratio_ins = float(raw_data['procs_ins'][trace])\
                                          / float(raw_data['procs_ins'][trace_list[0]])

                        mod_factors['inst_scale'][trace] = float(raw_data['useful_ins'][trace_list[0]]) \
                                                   / float(raw_data['useful_ins'][trace]) \
                                                           * procs_ratio_ins * 100.0
                else:
                    mod_factors['inst_scale'][trace] = 'Non-Avail'
            else:
                mod_factors['inst_scale'][trace] = 'Non-Avail'
        except:
            mod_factors['inst_scale'][trace] = 'NaN'

        # ins scale + Serial I/O
        try:  # except NaN
            if len(trace_list) > 1:
                if other_metrics['io_posix'][trace] > 0.0 or other_metrics['flushing'][trace] > 0.0:
                    useful_ins_plus_io_0 = float(raw_data['useful_ins'][trace_list[0]]) \
                                       + float(raw_data['io_ins'][trace_list[0]]) \
                                       + float(raw_data['flushing_ins'][trace_list[0]])
                    useful_ins_plus_io_n = float(raw_data['useful_ins'][trace]) \
                                       + float(raw_data['io_ins'][trace]) \
                                       + float(raw_data['flushing_ins'][trace])
                    if scaling == 'strong':
                        mod_factors_scale_plus_io['inst_scale'][trace] = float(useful_ins_plus_io_0) \
                                                                 / float(useful_ins_plus_io_n) * 100.0
                    else:
                        procs_ratio_ins = float(raw_data['procs_ins'][trace]) \
                                          / float(raw_data['procs_ins'][trace_list[0]])
                        mod_factors_scale_plus_io['inst_scale'][trace] = float(useful_ins_plus_io_0) \
                                                                         / float(useful_ins_plus_io_n)\
                                                                         * procs_ratio_ins * 100.0
                else:
                    mod_factors_scale_plus_io['inst_scale'][trace] = float(mod_factors['inst_scale'][trace])
            else:
                mod_factors_scale_plus_io['inst_scale'][trace] = 'Non-Avail'
        except:
            mod_factors_scale_plus_io['inst_scale'][trace] = 'NaN'

        ### Host Computational Metrics
        try:  # except NaN
            if outmpi_measures and (trace_mode[trace] == 'Detailed+MPI+CUDA') and (len(trace_list) > 1):
                if scaling == 'strong':
                    host_factors['host_comp_scale'][trace] = float(raw_data['useful_not_0_tot'][trace_list[0]]) \
                                                   / float(raw_data['useful_not_0_tot'][trace]) * 100.0
                else:
                    host_factors['host_comp_scale'][trace] = float(raw_data['useful_not_0_tot'][trace_list[0]]) \
                                                   / float(raw_data['useful_not_0_tot'][trace]) * proc_ratio * 100.0
                host_factors['host_ipc_scale'][trace] = mod_factors['ipc_scale'][trace]
                host_factors['host_inst_scale'][trace] = mod_factors['inst_scale'][trace]
                host_factors['host_freq_scale'][trace] = mod_factors['freq_scale'][trace]
            else:
                host_factors['host_comp_scale'][trace] = 'Non-Avail'
                host_factors['host_ipc_scale'][trace] = 'Non-Avail'
                host_factors['host_inst_scale'][trace] = 'Non-Avail'
                host_factors['host_freq_scale'][trace] = 'Non-Avail'
        except:
            host_factors['host_comp_scale'][trace] = 'NaN'
            host_factors['host_ipc_scale'][trace] = 'NaN'
            host_factors['host_inst_scale'][trace] = 'NaN'
            host_factors['host_freq_scale'][trace] = 'NaN'
        
        try:  # except NaN
            if outmpi_measures and (trace_mode[trace] == 'Detailed+MPI+CUDA'):
                if len(trace_list) > 1:
                    host_factors['host_global_eff'][trace] = (host_factors['host_parallel_eff'][trace]/100) \
                * (host_factors['host_comp_scale'][trace]/100) * 100
                else:
                    host_factors['host_global_eff'][trace] = host_factors['host_parallel_eff'][trace]
            else:
                host_factors['host_global_eff'][trace] = 'Non-Avail'  
        except:
            host_factors['host_global_eff'][trace] = 'NaN'

        try:  # except NaN
            if len(trace_list) > 1:
                if scaling == 'strong':
                    other_metrics['speedup'][trace] = float(raw_data['runtime'][trace_list[0]]
                                                            / raw_data['runtime'][trace])
                else:
                    other_metrics['speedup'][trace] = float(raw_data['runtime'][trace_list[0]]
                                                            / raw_data['runtime'][trace] * proc_ratio)
            else:
                other_metrics['speedup'][trace] = 'Non-Avail'
        except:
            other_metrics['speedup'][trace] = 'NaN'

        try:  # except NaN
            other_metrics['elapsed_time'][trace] = float(raw_data['runtime'][trace] * 0.000001)
        except:
            other_metrics['elapsed_time'][trace] = 'NaN'

        try:  # except NaN
            if len(trace_list) > 1:
                #other_metrics['efficiency'][trace] = other_metrics['speedup'][trace] / proc_ratio
                if scaling == 'strong':
                    other_metrics['efficiency'][trace] = raw_data['runtime'][trace_list[0]] \
                                                        / (raw_data['runtime'][trace] * proc_ratio)
                else:
                    other_metrics['efficiency'][trace] = raw_data['runtime'][trace_list[0]] \
                                                        / raw_data['runtime'][trace]
            else:
                other_metrics['efficiency'][trace] = 'Non-Avail'
        except:
            other_metrics['efficiency'][trace] = 'NaN'

    return mod_factors, mod_factors_scale_plus_io, hybrid_factors, other_metrics, device_factors, host_factors


def print_mod_factors_table(mod_factors, other_metrics, mod_factors_scale_plus_io, hybrid_factors,device_factors , trace_list,
                            trace_processes, trace_tasks, trace_threads, trace_mode, raw_data):
    """Prints the model factors table in human readable form on stdout."""
    global mod_factors_doc, mod_hybrid_factors_doc, mod_device_factors_doc

    warning_io = []
    warning_flush = []
    warning_flush_wrong = []
    warning_simulation = []
    for trace in trace_list:
        if 10.0 <= other_metrics['flushing'][trace] < 15.0:
            warning_flush.append(1)
        elif other_metrics['flushing'][trace] >= 15.0:
            warning_flush_wrong.append(1)
        if other_metrics['io_posix'][trace] >= 5.0:
            warning_io.append(1)
        if hybrid_factors['serial_eff'][trace] == "Warning!" \
                or hybrid_factors['serial_eff'][trace] == "Warning!":
            warning_simulation.append(1)

    if len(warning_flush_wrong) > 0:
        print("WARNING! Flushing in a trace is too high. Disabling standard output metrics...")
        print("         Flushing is an overhead due to the tracer, please review your trace.")
        print('')
        return

    count_mode = 0
    for trace in trace_list:
        if trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            count_mode += 1

    if count_mode == len(trace_list):
        del(mod_factors['serial_eff'])
        del (mod_factors['transfer_eff'])
        del(mod_factors_doc['serial_eff'])
        del (mod_factors_doc['transfer_eff'])

    # Update the hybrid parallelism mode
    trace_mode_doc = trace_mode[trace_list[0]]
    if trace_mode_doc[0:len("Detailed+MPI+")] == "Detailed+MPI+":
        mod_hybrid_factors_doc['omp_parallel_eff'] = "   -- " + \
                                                     trace_mode_doc[len("Detailed+MPI+"):] + " Parallel efficiency"
        mod_hybrid_factors_doc['omp_load_balance'] = "      -- " + \
                                                     trace_mode_doc[len("Detailed+MPI+"):] + " Load Balance"
        mod_hybrid_factors_doc['omp_comm_eff'] = "      -- " + \
                                                 trace_mode_doc[len("Detailed+MPI+"):] + " Communication efficiency"
    
    # print('')
    print('\n Overview of the Efficiency metrics:')

    longest_name = len(sorted(mod_hybrid_factors_doc.values(), key=len)[-1])
    
    if trace_mode[trace] == "Detailed+MPI+CUDA":
        line = '#Procs(ProcsxThreads)(#GPUs)[TraceOrder]'.rjust(longest_name)
    else:
        line = '#Procs(ProcsxThreads)[TraceOrder]'.rjust(longest_name)



    
    line_trace_mode = 'Trace mode'.rjust(longest_name)
    if len(trace_list) == 1:
        limit_min = trace_processes[trace_list[0]]
        limit_max = trace_processes[trace_list[0]]
    else:
        limit_min = trace_processes[trace_list[0]]
        limit_max = trace_processes[trace_list[len(trace_list)-1]]

    # To control same number of processes for the header on plots and table
    same_procs = True
    procs_trace_prev = trace_processes[trace_list[0]]
    tasks_trace_prev = trace_tasks[trace_list[0]]
    threads_trace_prev = trace_threads[trace_list[0]]

    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        if procs_trace_prev == trace_processes[trace] and tasks_trace_prev == tasks \
                and threads_trace_prev == threads:
            same_procs *= True
        else:
            same_procs *= False

    # BEGIN To adjust header to big number of processes
    procs_header = []
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        #if limit_min == limit_max and same_procs and len(trace_list) > 1:
        #    s_xtics = (str(trace_processes[trace]) + '[' + str(index+1) + ']')
        #else:
        if trace_mode[trace] == "Detailed+MPI+CUDA":
            s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')'+'(#'+ str(raw_data['count_devices'][trace]))+')'
        elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')')
        else:
            s_xtics = (str(trace_processes[trace]))
        #if s_xtics in procs_header:
        s_xtics += '[' + str(index + 1) + ']'
        procs_header.append(s_xtics)

    max_len_header = 0
    for proc_h in procs_header:
        if max_len_header < len(proc_h):
            max_len_header = len(proc_h)

    value_to_adjust = 14
    if max_len_header > value_to_adjust:
        value_to_adjust = max_len_header + 1
    # END To adjust header to big number of processes

    label_xtics = []
    label_trace_mode = []
    mode_string = ""
    for index, trace in enumerate(trace_list):
        line += ' | '
        line_trace_mode += ' | '
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        #if limit_min == limit_max and same_procs and len(trace_list) > 1:
        #    s_xtics = (str(trace_processes[trace]) + '[' + str(index+1) + ']')
        #else:
        if trace_mode[trace] == "Detailed+MPI+CUDA":
            s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')'+'(#'+ str(raw_data['count_devices'][trace]))+')'
        elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')')
        else:
            s_xtics = (str(trace_processes[trace]))
        #if s_xtics in label_xtics:
        s_xtics += '[' + str(index + 1) + ']'
        label_xtics.append(s_xtics)
        line += s_xtics.rjust(value_to_adjust)
        # For trace mode
        if trace_mode[trace][0:len("Detailed")] == "Detailed":
            mode_string = trace_mode[trace][len("Detailed+"):]
        elif trace_mode[trace][0:len("Burst")] == "Burst":
            mode_string = trace_mode[trace]
        elif trace_mode[trace] == "Sampling":
            mode_string = trace_mode[trace]
        line_trace_mode += mode_string.rjust(value_to_adjust)
        label_trace_mode.append(mode_string)

    print(''.ljust(len(line), '='))
    print(line_trace_mode)
    print(line)
    line_procs_factors = line

    print(''.ljust(len(line), '='))

    for mod_key in mod_factors_doc:
        line = mod_factors_doc[mod_key].ljust(longest_name)
        for trace in trace_list:
            line += ' | '
            try:  # except NaN
                line += ('{0:.2f}%'.format(mod_factors[mod_key][trace])).rjust(value_to_adjust)
            except ValueError:
                line += ('{}'.format(mod_factors[mod_key][trace])).rjust(value_to_adjust)
        print(line)

    io_metrics = []
    io_metrics.append(0)
    for mod_key in mod_factors_scale_plus_io_doc:
        for trace in trace_list:
            if mod_factors_scale_plus_io[mod_key][trace] != 0:
                io_metrics.append(1)

    if (len(warning_flush) >= 1 or len(warning_io) >= 1) and len(trace_list) > 1:
        print(''.ljust(len(line_procs_factors), '-'))
        for mod_key in mod_factors_scale_plus_io_doc:
            line = mod_factors_scale_plus_io_doc[mod_key].ljust(longest_name)
            for trace in trace_list:
                line += ' | '
                try:  # except NaN
                    line += ('{0:.2f}%'.format(mod_factors_scale_plus_io[mod_key][trace])).rjust(value_to_adjust)
                except ValueError:
                    line += ('{}'.format(mod_factors_scale_plus_io[mod_key][trace])).rjust(value_to_adjust)
            print(line)
        print(''.ljust(len(line_procs_factors), '='))
    else:
        print(''.ljust(len(line_procs_factors), '-'))
        
    for mod_key in mod_hybrid_factors_doc:
        line = mod_hybrid_factors_doc[mod_key].ljust(longest_name)
        for trace in trace_list:
            line += ' | '
            try:  # except NaN
                if str(hybrid_factors[mod_key][trace]) != 'nan':
                    line += ('{0:.2f}%'.format(hybrid_factors[mod_key][trace])).rjust(value_to_adjust)
                elif str(hybrid_factors[mod_key][trace]) == 'nan':
                    line += ('{}'.format('NaN')).rjust(value_to_adjust)
                else:
                    line += ('{}'.format(hybrid_factors[mod_key][trace])).rjust(value_to_adjust)
            except ValueError:
                line += ('{}'.format(hybrid_factors[mod_key][trace])).rjust(value_to_adjust)
        print(line)

#####
    print(''.ljust(len(line_procs_factors), '='))
    if len(warning_simulation) > 0:
        print("===> Warning! Metrics obtained from simulated traces exceed 100%. "
              "Please review original and simulated traces.")
    print('')

#### TALP metrics
def print_mod_factors_table_talp(mod_factors, other_metrics, mod_factors_scale_plus_io, hybrid_factors,device_factors, host_factors , trace_list,
                            trace_processes, trace_tasks, trace_threads, trace_mode, raw_data):
    """Prints the model factors table in human readable form on stdout."""
    global mod_factors_doc, mod_hybrid_factors_doc, mod_device_factors_doc,mod_host_factors_doc

    warning_io = []
    warning_flush = []
    warning_flush_wrong = []
    warning_simulation = []
    for trace in trace_list:
        if 10.0 <= other_metrics['flushing'][trace] < 15.0:
            warning_flush.append(1)
        elif other_metrics['flushing'][trace] >= 15.0:
            warning_flush_wrong.append(1)
        if other_metrics['io_posix'][trace] >= 5.0:
            warning_io.append(1)
        if hybrid_factors['serial_eff'][trace] == "Warning!" \
                or hybrid_factors['serial_eff'][trace] == "Warning!":
            warning_simulation.append(1)

    if len(warning_flush_wrong) > 0:
        print("WARNING! Flushing in a trace is too high. Disabling standard output metrics...")
        print("         Flushing is an overhead due to the tracer, please review your trace.")
        print('')
        return

    count_mode = 0
    for trace in trace_list:
        if trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            count_mode += 1

    if count_mode == len(trace_list):
        del(mod_factors['serial_eff'])
        del (mod_factors['transfer_eff'])
        del(mod_factors_doc['serial_eff'])
        del (mod_factors_doc['transfer_eff'])

    # Update the hybrid parallelism mode
    trace_mode_doc = trace_mode[trace_list[0]]
    if trace_mode_doc[0:len("Detailed+MPI+")] == "Detailed+MPI+":
        mod_hybrid_factors_doc['omp_parallel_eff'] = "   -- " + \
                                                     trace_mode_doc[len("Detailed+MPI+"):] + " Parallel efficiency"
        mod_hybrid_factors_doc['omp_load_balance'] = "      -- " + \
                                                     trace_mode_doc[len("Detailed+MPI+"):] + " Load Balance"
        mod_hybrid_factors_doc['omp_comm_eff'] = "      -- " + \
                                                 trace_mode_doc[len("Detailed+MPI+"):] + " Communication efficiency"
    
    # print('')
    print('\n Overview of the Efficiency metrics:')

    longest_name = len(sorted(mod_hybrid_factors_doc.values(), key=len)[-1])
    
    if trace_mode[trace] == "Detailed+MPI+CUDA":
        line = '#Procs(ProcsxThreads)(#GPUs)[TraceOrder]'.rjust(longest_name)
    else:
        line = '#Procs(ProcsxThreads)[TraceOrder]'.rjust(longest_name)

    
    line_trace_mode = 'Trace mode'.rjust(longest_name)
    if len(trace_list) == 1:
        limit_min = trace_processes[trace_list[0]]
        limit_max = trace_processes[trace_list[0]]
    else:
        limit_min = trace_processes[trace_list[0]]
        limit_max = trace_processes[trace_list[len(trace_list)-1]]

    # To control same number of processes for the header on plots and table
    same_procs = True
    procs_trace_prev = trace_processes[trace_list[0]]
    tasks_trace_prev = trace_tasks[trace_list[0]]
    threads_trace_prev = trace_threads[trace_list[0]]

    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        if procs_trace_prev == trace_processes[trace] and tasks_trace_prev == tasks \
                and threads_trace_prev == threads:
            same_procs *= True
        else:
            same_procs *= False

    # BEGIN To adjust header to big number of processes
    procs_header = []
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        #if limit_min == limit_max and same_procs and len(trace_list) > 1:
        #    s_xtics = (str(trace_processes[trace]) + '[' + str(index+1) + ']')
        #else:
        if trace_mode[trace] == "Detailed+MPI+CUDA":
            s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')'+'(#'+ str(raw_data['count_devices'][trace]))+')'
        elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')')
        else:
            s_xtics = (str(trace_processes[trace]))
        #if s_xtics in procs_header:
        s_xtics += '[' + str(index + 1) + ']'
        procs_header.append(s_xtics)

    max_len_header = 0
    for proc_h in procs_header:
        if max_len_header < len(proc_h):
            max_len_header = len(proc_h)

    value_to_adjust = 14
    if max_len_header > value_to_adjust:
        value_to_adjust = max_len_header + 1
    # END To adjust header to big number of processes

    label_xtics = []
    label_trace_mode = []
    mode_string = ""
    for index, trace in enumerate(trace_list):
        line += ' | '
        line_trace_mode += ' | '
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        #if limit_min == limit_max and same_procs and len(trace_list) > 1:
        #    s_xtics = (str(trace_processes[trace]) + '[' + str(index+1) + ']')
        #else:
        if trace_mode[trace] == "Detailed+MPI+CUDA":
            s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')'+'(#'+ str(raw_data['count_devices'][trace]))+')'
        elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')')
        else:
            s_xtics = (str(trace_processes[trace]))
        #if s_xtics in label_xtics:
        s_xtics += '[' + str(index + 1) + ']'
        label_xtics.append(s_xtics)
        line += s_xtics.rjust(value_to_adjust)
        # For trace mode
        if trace_mode[trace][0:len("Detailed")] == "Detailed":
            mode_string = trace_mode[trace][len("Detailed+"):]
        elif trace_mode[trace][0:len("Burst")] == "Burst":
            mode_string = trace_mode[trace]
        elif trace_mode[trace] == "Sampling":
            mode_string = trace_mode[trace]
        line_trace_mode += mode_string.rjust(value_to_adjust)
        label_trace_mode.append(mode_string)

    print(''.ljust(len(line), '='))
    print(line_trace_mode)
    print(line)
    line_procs_factors = line

    print(''.ljust(len(line), '='))

    io_metrics = []
    io_metrics.append(0)
    for mod_key in mod_factors_scale_plus_io_doc:
        for trace in trace_list:
            if mod_factors_scale_plus_io[mod_key][trace] != 0:
                io_metrics.append(1)

    if (len(warning_flush) >= 1 or len(warning_io) >= 1) and len(trace_list) > 1:
        print(''.ljust(len(line_procs_factors), '-'))
        for mod_key in mod_factors_scale_plus_io_doc:
            line = mod_factors_scale_plus_io_doc[mod_key].ljust(longest_name)
            for trace in trace_list:
                line += ' | '
                try:  # except NaN
                    line += ('{0:.2f}%'.format(mod_factors_scale_plus_io[mod_key][trace])).rjust(value_to_adjust)
                except ValueError:
                    line += ('{}'.format(mod_factors_scale_plus_io[mod_key][trace])).rjust(value_to_adjust)
            print(line)
        print(''.ljust(len(line_procs_factors), '='))
    else:
        print(''.ljust(len(line_procs_factors), '-'))
        
    for mod_key in mod_host_factors_doc:
        line = mod_host_factors_doc[mod_key].ljust(longest_name)
        for trace in trace_list:
            line += ' | '
            try:  # except NaN
                if str(host_factors[mod_key][trace]) != 'nan':
                    line += ('{0:.2f}%'.format(host_factors[mod_key][trace])).rjust(value_to_adjust)
                elif str(host_factors[mod_key][trace]) == 'nan':
                    line += ('{}'.format('NaN')).rjust(value_to_adjust)
                else:
                    line += ('{}'.format(host_factors[mod_key][trace])).rjust(value_to_adjust)
            except ValueError:
                line += ('{}'.format(host_factors[mod_key][trace])).rjust(value_to_adjust)
        print(line)
## TO Devices metrics
    print(''.ljust(len(line_procs_factors), '-'))
    for mod_key in mod_device_factors_doc:
        line = mod_device_factors_doc[mod_key].ljust(longest_name)
        for trace in trace_list:
            line += ' | '
            try:  # except NaN
                if str(device_factors[mod_key][trace]) != 'nan':
                    line += ('{0:.2f}%'.format(device_factors[mod_key][trace])).rjust(value_to_adjust)
                elif str(device_factors[mod_key][trace]) == 'nan':
                    line += ('{}'.format('NaN')).rjust(value_to_adjust)
                else:
                    line += ('{}'.format(device_factors[mod_key][trace])).rjust(value_to_adjust)
            except ValueError:
                line += ('{}'.format(device_factors[mod_key][trace])).rjust(value_to_adjust)
        print(line)    

#####
    print(''.ljust(len(line_procs_factors), '='))
    if len(warning_simulation) > 0:
        print("===> Warning! Metrics obtained from simulated traces exceed 100%. "
              "Please review original and simulated traces.")
    print('')


def print_other_metrics_table(other_metrics, trace_list, trace_processes, trace_tasks, trace_threads, trace_mode):
    """Prints the other metrics table in human readable form on stdout."""
    global other_metrics_doc

    print('Overview of the Efficiency, Speedup, IPC and Frequency:')

    longest_name = len(sorted(other_metrics_doc.values(), key=len)[-1])

    line = ''.rjust(longest_name)
    if len(trace_list) == 1:
        limit_min = trace_processes[trace_list[0]]
        limit_max = trace_processes[trace_list[0]]
    else:
        limit_min = trace_processes[trace_list[0]]
        limit_max = trace_processes[trace_list[len(trace_list)-1]]

    # To control same number of processes for the header on plots and table
    same_procs = True
    procs_trace_prev = trace_processes[trace_list[0]]
    tasks_trace_prev = trace_tasks[trace_list[0]]
    threads_trace_prev = trace_threads[trace_list[0]]
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        if procs_trace_prev == trace_processes[trace] and tasks_trace_prev == tasks \
                and threads_trace_prev == threads:
            same_procs *= True
        else:
            same_procs *= False

    # BEGIN To adjust header to big number of processes
    procs_header = []
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        s_xtics = (str(trace_processes[trace]))
        #if limit_min == limit_max and same_procs and len(trace_list) > 1:
        #    s_xtics = (str(trace_processes[trace]) + '[' + str(index+1) + ']')
        #else:
        if trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')')
        else:
            s_xtics = (str(trace_processes[trace]))
        #if s_xtics in procs_header:
        s_xtics += '[' + str(index + 1) + ']'
        procs_header.append(s_xtics)

    max_len_header = 0
    for proc_h in procs_header:
        if max_len_header < len(proc_h):
            max_len_header = len(proc_h)

    value_to_adjust = 10
    if max_len_header > value_to_adjust:
        value_to_adjust = max_len_header + 1
    # END To adjust header to big number of processes
    s_xtics = ''
    label_xtics = []
    for index, trace in enumerate(trace_list):
        line += ' | '
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        s_xtics = (str(trace_processes[trace]))
        #if limit_min == limit_max and same_procs and len(trace_list) > 1:
        #    s_xtics = (str(trace_processes[trace]) + '[' + str(index+1) + ']')
        #else:
        if trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')')
        else:
            s_xtics = (str(trace_processes[trace]))
        #if s_xtics in procs_header:
        s_xtics += '[' + str(index + 1) + ']'
        label_xtics.append(s_xtics)
        line += s_xtics.rjust(value_to_adjust)

    print(''.ljust(len(line), '-'))
    print(line)
    line_head = line
    print(''.ljust(len(line), '-'))

    for mod_key in other_metrics_doc:
        line = other_metrics_doc[mod_key].ljust(longest_name)
        if len(trace_list) > 1:
            if mod_key in ['speedup', 'ipc', 'freq', 'elapsed_time', 'efficiency']:
                for trace in trace_list:
                    line += ' | '
                    try:  # except NaN
                        line += ('{0:.2f}'.format(other_metrics[mod_key][trace])).rjust(value_to_adjust)
                    except ValueError:
                        line += ('{}'.format(other_metrics[mod_key][trace])).rjust(value_to_adjust)
                print(line)
        else:
            if mod_key in ['ipc', 'freq', 'elapsed_time']:
                for trace in trace_list:
                    line += ' | '
                    try:  # except NaN
                        line += ('{0:.2f}'.format(other_metrics[mod_key][trace])).rjust(value_to_adjust)
                    except ValueError:
                        line += ('{}'.format(other_metrics[mod_key][trace])).rjust(value_to_adjust)
                print(line)
    print(''.ljust(len(line_head), '-'))
    # print('')

    warning_io = []
    warning_flush = []
    for trace in trace_list:
        if other_metrics['flushing'][trace] >= 10.0:
            warning_flush.append(1)
        if other_metrics['io_mpiio'][trace] >= 5.0 or other_metrics['io_posix'][trace] >= 5.0:
            warning_io.append(1)

    for mod_key in other_metrics_doc:
        line = other_metrics_doc[mod_key].ljust(longest_name)
        # Print empty line to separate values
        if mod_key in ['freq'] and len(warning_flush) > 0:
            print('')
            print("Overview of tracer\'s flushing weight:")
            print(''.ljust(len(line_head), '-'))

        if mod_key not in ['speedup', 'ipc', 'freq', 'elapsed_time', 'efficiency']:
            if mod_key in ['flushing']:
                for trace in trace_list:
                    line += ' | '
                    try:  # except NaN
                        line += ('{0:.2f}%'.format(other_metrics[mod_key][trace])).rjust(value_to_adjust)
                    except ValueError:
                        line += ('{}'.format(other_metrics[mod_key][trace])).rjust(value_to_adjust)
                if len(warning_flush) > 0:
                    print(line)
                    print(''.ljust(len(line_head), '-'))
            elif mod_key in ['io_mpiio','io_posix','io_eff']:
                for trace in trace_list:
                    line += ' | '
                    try:  # except NaN
                        line += ('{0:.2f}%'.format(other_metrics[mod_key][trace])).rjust(value_to_adjust)
                    except ValueError:
                        line += ('{}'.format(other_metrics[mod_key][trace])).rjust(value_to_adjust)
                if len(warning_io) > 0:
                    print(line)

        # Print headers I/O
        if mod_key in ['flushing'] and len(warning_io) > 0:
            print(''.ljust(len(line), ' '))
            print('Overview of File I/O weight:')
            print(''.ljust(len(line), '-'))
        if mod_key in ['io_eff'] and len(warning_io) > 0:
            print(''.ljust(len(line), '-'))
            # print('')

    if len(warning_flush) > 0:
        message_warning_flush = "WARNING! %Flushing is high and affects computation of efficiency metrics."
    else:
        message_warning_flush = ""
    if len(warning_io) > 0:
        message_warning_io = "WARNING! % File I/O is high and affects computation of efficiency metrics."
    else:
        message_warning_io = ""
    print(message_warning_flush+message_warning_io)
    # print('')


def print_efficiency_table(mod_factors, hybrid_factors, trace_list, trace_processes, trace_tasks, trace_threads, trace_mode):
    """Prints the model factors table in human readable form on stdout."""
    global mod_factors_doc, mod_hybrid_factors_doc

    delimiter = ','
    # To control same number of processes for the header on plots and table
    same_procs = True
    procs_trace_prev = trace_processes[trace_list[0]]
    tasks_trace_prev = trace_tasks[trace_list[0]]
    threads_trace_prev = trace_threads[trace_list[0]]
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        if procs_trace_prev == trace_processes[trace] and tasks_trace_prev == tasks \
                and threads_trace_prev == threads:
            same_procs *= True
        else:
            same_procs *= False

    if len(trace_list) == 1:
        limit_min = trace_processes[trace_list[0]]
        limit_max = trace_processes[trace_list[0]]
    else:
        limit_min = trace_processes[trace_list[0]]
        limit_max = trace_processes[trace_list[len(trace_list) - 1]]

    file_path = os.path.join(os.getcwd(), 'efficiency_table_global.csv')
    with open(file_path, 'w') as output:

        line = '\"Number of processes\"'
        label_xtics = []
        for index, trace in enumerate(trace_list):
            line += delimiter
            tasks = trace_tasks[trace]
            threads = trace_threads[trace]

            if trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')')
            else:
                s_xtics = (str(trace_processes[trace]))
            #if s_xtics in label_xtics:
            s_xtics += '[' + str(index + 1) + ']'
            label_xtics.append(s_xtics)
            line += s_xtics

        output.write(line + '\n')

        for mod_key in mod_factors_doc:
            if mod_key not in ['speedup', 'ipc', 'freq', 'elapsed_time', 'efficiency', 'flushing', 'io_mpiio', 'io_posix']:
                if mod_key in ['parallel_eff', 'comp_scale']:
                    line = "\"" + mod_factors_doc[mod_key].replace('  ', '', 2) + "\""
                elif mod_key in ['load_balance', 'comm_eff','ipc_scale', 'inst_scale','freq_scale']:
                    line = "\"" + mod_factors_doc[mod_key].replace('     ', '', 2) + "\""
                else:
                    line = "\"" + mod_factors_doc[mod_key] + "\""
                for trace in trace_list:
                    line += delimiter
                    try:  # except NaN
                        if mod_factors[mod_key][trace] == "Non-Avail" or mod_factors[mod_key][trace] == "Warning!":
                            line += '0.00'
                        else:
                            line += '{0:.2f}'.format(mod_factors[mod_key][trace])
                    except ValueError:
                        line += '{}'.format(mod_factors[mod_key][trace])
                output.write(line + '\n')

        # Create Gnuplot file for efficiency plot
        gp_template = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'cfgs', 'efficiency_table_global.gp')
        content = []

        with open(gp_template) as f:
            content = f.readlines()

        limit_procs = 500 + len(trace_list) * 65
        ## print(limit_procs)

        # Replace xrange
        content = [line.replace('#REPLACE_BY_SIZE', ''.join(['set terminal pngcairo enhanced dashed crop size ',
                                                             str(limit_procs), ',460 font "Latin Modern Roman,14"']))
                   for line in content]

        file_path = os.path.join(os.getcwd(), 'efficiency_table_global.gp')
        with open(file_path, 'w') as f:
            f.writelines(content)
        # print('======== Plot (gnuplot File): EFFICIENCY Table ========')
        if len(trace_list) > 1:
            print('Global Efficiency Table written to ' + file_path[:len(file_path) - 3] + '.gp')
        # print('')

    delimiter = ','
    file_path = os.path.join(os.getcwd(), 'efficiency_table_hybrid.csv')
    with open(file_path, 'w') as output:
        line = '\"Number of processes\"'
        label_xtics = []
        for index, trace in enumerate(trace_list):
            line += delimiter
            tasks = trace_tasks[trace]
            threads = trace_threads[trace]
            #if limit_min == limit_max and same_procs and len(trace_list) > 1:
            #    s_xtics = (str(trace_processes[trace]) + '[' + str(index + 1) + ']')
            #else:
            if trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                s_xtics = (str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')')
            else:
                s_xtics = (str(trace_processes[trace]))
            #if s_xtics in label_xtics:
            s_xtics += '[' + str(index + 1) + ']'
            label_xtics.append(s_xtics)
            line += s_xtics
        output.write(line + '\n')

        for mod_key in mod_hybrid_factors_doc:
            if mod_key in ['mpi_parallel_eff', 'omp_parallel_eff']:
                line = "\"" + mod_hybrid_factors_doc[mod_key].replace('     ', '', 2) + "\""
            elif mod_key in ['mpi_load_balance', 'mpi_comm_eff','omp_load_balance', 'omp_comm_eff']:
                line = "\"" + mod_hybrid_factors_doc[mod_key].replace('       ', '', 2) + "\""
            elif mod_key in ['serial_eff', 'transfer_eff']:
                line = "\"" + mod_hybrid_factors_doc[mod_key].replace('         ', '          ', 2) + "\""
            else:
                line = "\"" + mod_hybrid_factors_doc[mod_key] + "\""
            for trace in trace_list:
                line += delimiter
                try:  # except NaN
                    if hybrid_factors[mod_key][trace] == "Non-Avail" or hybrid_factors[mod_key][trace] == "Warning!":
                        line += '0.00'
                    else:
                        line += '{0:.2f}'.format(hybrid_factors[mod_key][trace])
                except ValueError:
                    line += '{}'.format(hybrid_factors[mod_key][trace])
            output.write(line + '\n')

        # Create Gnuplot file for efficiency plot
        gp_template = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'cfgs', 'efficiency_table_hybrid.gp')
        content = []

        with open(gp_template) as f:
            content = f.readlines()

        limit_procs = 510 + len(trace_list) * 80

        # Replace xrange
        content = [line.replace('#REPLACE_BY_SIZE', ''.join(['set terminal pngcairo enhanced dashed crop size ',
                                                             str(limit_procs), ',460 font "Latin Modern Roman,14"']))
                   for line in content]

        file_path = os.path.join(os.getcwd(), 'efficiency_table_hybrid.gp')
        with open(file_path, 'w') as f:
            f.writelines(content)
        # print('======== Plot (gnuplot File): EFFICIENCY Table ========')
        if len(trace_list) > 1:
            print('Hybrid Efficiency Table written to ' + file_path[:len(file_path) - 3] + '.gp')
        #print('')


def print_mod_factors_csv(mod_factors, hybrid_factors, trace_list, trace_processes):
    """Prints the model factors table in a csv file."""
    global mod_factors_doc, mod_hybrid_factors_doc

    delimiter = ';'
    # File is stored in the trace directory
    # file_path = os.path.join(os.path.dirname(os.path.realpath(trace_list[0])), 'modelfactors.csv')
    # File is stored in the execution directory
    file_path = os.path.join(os.getcwd(), 'modelfactors.csv')

    with open(file_path, 'w') as output:
        line = "\"Number of processes\""
        for trace in trace_list:
            line += delimiter
            line += str(trace_processes[trace])
        output.write(line + '\n')

        for mod_key in mod_factors_doc:
            line = "\"" + mod_factors_doc[mod_key].replace('  ', '', 2) + "\""
            for trace in trace_list:
                line += delimiter
                try:  # except NaN
                    line += '{0:.6f}'.format(mod_factors[mod_key][trace])
                except ValueError:
                    line += '{}'.format(mod_factors[mod_key][trace])
            output.write(line + '\n')

        for mod_key in mod_hybrid_factors_doc:
            line = "\"" + mod_hybrid_factors_doc[mod_key].replace('  ', '', 2) + "\""
            for trace in trace_list:
                line += delimiter
                try:  # except NaN
                    line += '{0:.6f}'.format(hybrid_factors[mod_key][trace])
                except ValueError:
                    line += '{}'.format(hybrid_factors[mod_key][trace])
            output.write(line + '\n')

        output.write('#\n')

    print('======== Output Files: Metrics and Plots  ========')
    print('Model factors written to ' + file_path)


def print_other_metrics_csv(other_metrics, trace_list, trace_processes):
    """Prints the model factors table in a csv file."""
    global other_metrics_doc

    delimiter = ';'
    # File is stored in the trace directory
    # file_path = os.path.join(os.path.dirname(os.path.realpath(trace_list[0])), 'modelfactors.csv')
    # File is stored in the execution directory
    file_path = os.path.join(os.getcwd(), 'other_metrics.csv')

    with open(file_path, 'w') as output:
        line = 'Number of processes'
        for trace in trace_list:
            line += delimiter
            line += str(trace_processes[trace])
        output.write(line + '\n')

        for mod_key in other_metrics_doc:
            line = other_metrics_doc[mod_key].replace('  ', '', 2)
            for trace in trace_list:
                line += delimiter
                try:  # except NaN
                    line += '{0:.6f}'.format(other_metrics[mod_key][trace])
                except ValueError:
                    line += '{}'.format(other_metrics[mod_key][trace])
            output.write(line + '\n')

        output.write('#\n')

    print('')
    print('======== Output File: Other Metrics ========')
    print('Speedup, IPC, Frequency, I/O and Flushing written to ' + file_path)


def plots_efficiency_table_matplot(trace_list, trace_processes, trace_tasks, trace_threads, trace_mode, cmdl_args):
    # Plotting using python
    # For plotting using python, read the csv file

    file_path = os.path.join(os.getcwd(), 'efficiency_table_hybrid.csv')
    df = pd.read_csv(file_path)
    metrics = df['Number of processes'].tolist()

    ## remove complete nan columns and put to DF1
    df1 = df.dropna(axis='columns', how='all')

    traces_procs = list(df.keys())[1:]
    # To control same number of processes for the header on plots and table
    same_procs = True
    procs_trace_prev = trace_processes[trace_list[0]]
    tasks_trace_prev = trace_tasks[trace_list[0]]
    threads_trace_prev = trace_threads[trace_list[0]]
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        if procs_trace_prev == trace_processes[trace] and tasks_trace_prev == tasks \
                and threads_trace_prev == threads:
            same_procs *= True
        else:
            same_procs *= False

    # Set limit for projection
    if cmdl_args.limit:
        limit = cmdl_args.limit
    else:
        limit = str(trace_processes[trace_list[len(trace_list)-1]])

    limit_min = trace_processes[trace_list[0]]

    # To xticks label
    label_xtics = []
    label_xtics_hybrid = []
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        #if int(limit) == int(limit_min) and same_procs:
        #    s_xtics = str(trace_processes[trace]) + '[' + str(index + 1) + ']'
        #else:
        if trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            s_xtics = str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')' \
                          + '[' + str(index + 1) + ']'
            label_xtics_hybrid.append(s_xtics)
        else:
            s_xtics = str(trace_processes[trace]) + '[' + str(index + 1) + ']'
        label_xtics.append(s_xtics)
    ##### End xticks

    # BEGIN To adjust header to big number of processes
    max_len_header = 7
    for labelx in label_xtics:
        if len(labelx) > max_len_header:
            max_len_header = len(labelx)

    # END To adjust header to big number of processes

    list_data = []

    for index, rows in df1.iterrows():
        list_temp = []
        for value in list(rows)[1:]:
            if pd.isna(value) or value == 0.0:
                list_temp.append(np.nan)
            else:
                list_temp.append(float(value))
        list_data.append(list_temp)

    list_np = np.array(list_data)

    idx = metrics
    #cols = label_xtics
    cols = label_xtics_hybrid
    # cols = traces_procs
    df = pd.DataFrame(list_np, index=idx, columns=cols)

    # min for 1 trace is x=3 for the (x,y) in figsize
    size_figure_y = len(idx) * 0.40
    size_figure_x = len(cols) * 0.14 * max_len_header
    plt.figure(figsize=(size_figure_x, size_figure_y))

    ax = sns.heatmap(df, cmap='RdYlGn', linewidths=0.05, annot=True, vmin=0, vmax=100, center=75, \
                     fmt='.2f', annot_kws={"size": 10}, cbar_kws={'label': 'Percentage(%)'})
    ## to align ylabels to left
    plt.yticks(rotation=0, ha='left')

    ax.xaxis.tick_top()
    # to adjust metrics
    len_pad = 0
    for metric in metrics:
        if len(metric) > len_pad:
            len_pad = len(metric)

    ax.yaxis.set_tick_params(pad=len_pad + 164)

    plt.savefig('efficiency_table_hybrid_matplot.png', bbox_inches='tight')

    # General Metrics plot

    file_path = os.path.join(os.getcwd(), 'efficiency_table_global.csv')
    df = pd.read_csv(file_path)
    #print(df)
    metrics = df['Number of processes'].tolist()

    traces_procs = list(df.keys())[1:]

    list_data = []
    for index, rows in df.iterrows():
        list_temp = []
        for value in list(rows)[1:]:
            if float(value) == 0.0:
                list_temp.append(np.nan)
            else:
                list_temp.append(float(value))
        # print(list_temp)
        list_data.append(list_temp)

    list_np = np.array(list_data)

    idx = metrics
    cols = label_xtics
    # cols = traces_procs

    df = pd.DataFrame(list_np, index=idx, columns=cols)

    # min for 1 traces is x=3 for the (x,y) in figsize
    size_figure_y = len(idx) * 0.40
    size_figure_x = len(cols) * 0.14 * max_len_header
    plt.figure(figsize=(size_figure_x, size_figure_y))

    ax = sns.heatmap(df, cmap='RdYlGn', linewidths=0.05, annot=True, vmin=0, vmax=100, center=75, \
                     fmt='.2f', annot_kws={"size": 10}, cbar_kws={'label': 'Percentage(%)'})
    ## to align ylabels to left
    plt.yticks(rotation=0, ha='left')
    ax.xaxis.tick_top()
    # to adjust metrics
    len_pad = 0
    for metric in metrics:
        if len(metric) > len_pad:
            len_pad = len(metric)

    ax.yaxis.set_tick_params(pad=len_pad + 140)

    plt.savefig('efficiency_table_global_matplot.png', bbox_inches='tight')


def plots_modelfactors_matplot(trace_list, trace_processes,trace_tasks, trace_threads, trace_mode, cmdl_args):
    # Plotting using python
    # For plotting using python, read the csv file
    count_mode = 0
    for trace in trace_list:
        if trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            count_mode += 1

    file_path = os.path.join(os.getcwd(), 'modelfactors.csv')
    df = pd.read_csv(file_path, sep=';')

    traces_procs = list(df.keys())[1:]

    if count_mode != len(trace_list):
        df_hybrid = df[10:19].dropna(axis='columns', how='all')
    else:
        df_hybrid = df[8:17].dropna(axis='columns', how='all')

    traces_procs_hybrid = list(df_hybrid.keys())[1:]

    #put all data in a list of list
    list_data = []
    for index, rows in df.iterrows():
        list_temp = []
        for value in list(rows)[1:]:
            if value != 'Non-Avail' and value != 'Warning!' and not pd.isna(value):
                list_temp.append(float(value))
            else:
                list_temp.append(0.0)

        list_data.append(list_temp)

    #put hybrid data in a list of list
    list_data_hybrid = []
    for index, rows in df_hybrid.iterrows():
        list_temp = []
        for value in list(rows)[1:]:
            if value != 'Non-Avail' and value != 'Warning!' and not pd.isna(value):
                list_temp.append(float(value))
            else:
                list_temp.append(0.0)

        list_data_hybrid.append(list_temp)

    # To control same number of processes for the header on plots and table
    same_procs = True
    procs_trace_prev = trace_processes[trace_list[0]]
    tasks_trace_prev = trace_tasks[trace_list[0]]
    threads_trace_prev = trace_threads[trace_list[0]]
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        if procs_trace_prev == trace_processes[trace] and tasks_trace_prev == tasks \
                and threads_trace_prev == threads:
            same_procs *= True
        else:
            same_procs *= False

    # Set limit for projection
    if cmdl_args.limit:
        limit = cmdl_args.limit
    else:
        limit = str(trace_processes[trace_list[len(trace_list)-1]])

    limit_min = trace_processes[trace_list[0]]

    # To xticks label
    label_xtics = []
    label_xtics_hybrid = []
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        if int(limit) == int(limit_min) and same_procs:
            s_xtics = str(trace_processes[trace]) + '[' + str(index + 1) + ']'
        else:
            if trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                s_xtics = str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')'
                label_xtics_hybrid.append(s_xtics+'[' + str(index + 1) + ']')
            else:
                s_xtics = str(trace_processes[trace])
        #if s_xtics in label_xtics:
        s_xtics += '[' + str(index + 1) + ']'
        label_xtics.append(s_xtics)

    ## Global Metrics
    plt.figure()

    if count_mode == len(trace_list):
        max_global = max([max(list_data[0]), max(list_data[1]), max(list_data[2]),
                      max(list_data[3]), max(list_data[4])])
        plt.plot(traces_procs, list_data[0], 'o-', color='black', label='Global Efficiency')
        plt.plot(traces_procs, list_data[1], 's--', color='magenta', label='Parallel Efficiency')
        plt.plot(traces_procs, list_data[2], '*:', color='red', label='Load Balance')
        plt.plot(traces_procs, list_data[3], 'x-.', color='green', label='Communication efficiency')
        plt.plot(traces_procs, list_data[4], 'v--', color='blue', label='Computation scalability')
    else:
        max_global = max([max(list_data[0]), max(list_data[1]), max(list_data[2]),
                          max(list_data[3]), max(list_data[6])])
        plt.plot(traces_procs, list_data[0], 'o-', color='black', label='Global Efficiency')
        plt.plot(traces_procs, list_data[1], 's--', color='magenta', label='Parallel Efficiency')
        plt.plot(traces_procs, list_data[2], '*:', color='red', label='Load Balance')
        plt.plot(traces_procs, list_data[3], 'x-.', color='green', label='Communication efficiency')
        plt.plot(traces_procs, list_data[6], 'v--', color='blue', label='Computation scalability')

    plt.xlabel("Number of Processes")
    plt.ylabel("Efficiency (%)")
    plt.xticks(tuple(traces_procs), tuple(label_xtics))
    # print(max_global)
    if float(max_global) < 100:
        max_global = 100

    plt.ylim(0, max_global+5)
    plt.legend()
    plt.savefig('modelfactors_global_matplot.png', bbox_inches='tight')

    ## Scale Metrics
    plt.figure()

    if count_mode == len(trace_list):
        max_scale = max([max(list_data[4]), max(list_data[5]), max(list_data[6]), max(list_data[7])])
        plt.plot(traces_procs, list_data[4], 'v-', color='blue', markerfacecolor='blue', label='Computation scalability')
        plt.plot(traces_procs, list_data[5], 'v--', color='skyblue', label='IPC scalability')
        plt.plot(traces_procs, list_data[6], 'v:', color='gray', label='Instruction scalability')
        plt.plot(traces_procs, list_data[7], 'v-.', color='darkviolet', label='Frequency scalability')
    else:
        max_scale = max([max(list_data[6]), max(list_data[7]), max(list_data[8]), max(list_data[9])])
        plt.plot(traces_procs, list_data[6], 'v-', color='blue', markerfacecolor='blue', label='Computation scalability')
        plt.plot(traces_procs, list_data[7], 'v--', color='skyblue', label='IPC scalability')
        plt.plot(traces_procs, list_data[8], 'v:', color='gray', label='Instruction scalability')
        plt.plot(traces_procs, list_data[9], 'v-.', color='darkviolet', label='Frequency scalability')

    plt.xlabel("Number of Processes")
    plt.ylabel("Efficiency (%)")
    plt.xticks(tuple(traces_procs), tuple(label_xtics))
    if float(max_scale) < 100:
        max_scale = 100

    plt.ylim(0, max_scale+5)
    plt.legend()
    plt.savefig('modelfactors_scale_matplot.png', bbox_inches='tight')

    ## Hybrid Metrics

    plt.figure()

    if count_mode == len(trace_list):
        max_hybrid = max([max(list_data[8]), max(list_data[9]), max(list_data[10]), max(list_data[11]),
                     max(list_data[14]), max(list_data[15]), max(list_data[16])])
        plt.plot(traces_procs, list_data[8], 's-', color='purple', label='Hybrid Parallel efficiency')
        plt.plot(traces_procs, list_data[9], 'o--', color='green', label='MPI Parallel efficiency')
        plt.plot(traces_procs, list_data[10], 'x-.', color='lime', label='MPI Load balance')
        plt.plot(traces_procs, list_data[11], '*:', color='lightseagreen', label='MPI Communication efficiency')
        plt.plot(traces_procs, list_data[14], 'h--', color='red', label='OpenMP Parallel efficiency')
        plt.plot(traces_procs, list_data[15], 'v-.', color='orange', label='OpenMP Load Balance')
        plt.plot(traces_procs, list_data[16], 'X:', color='salmon', label='OpenMP Communication efficiency')
    else:
        max_hybrid = max([max(list_data_hybrid[0]), max(list_data_hybrid[1]), max(list_data_hybrid[2]),
                          max(list_data_hybrid[3]), max(list_data_hybrid[4]), max(list_data_hybrid[5]),
                          max(list_data_hybrid[6])])
        plt.plot(traces_procs_hybrid, list_data_hybrid[0], 's-', color='purple', label='Hybrid Parallel efficiency')
        plt.plot(traces_procs_hybrid, list_data_hybrid[1], 'o--', color='green', label='MPI Parallel efficiency')
        plt.plot(traces_procs_hybrid, list_data_hybrid[2], 'x-.', color='lime', label='MPI Load balance')
        plt.plot(traces_procs_hybrid, list_data_hybrid[3], '*:', color='lightseagreen', label='MPI Communication efficiency')
        plt.plot(traces_procs_hybrid, list_data_hybrid[4], 'h--', color='red', label='OpenMP Parallel efficiency')
        plt.plot(traces_procs_hybrid, list_data_hybrid[5], 'v-.', color='orange', label='OpenMP Load Balance')
        plt.plot(traces_procs_hybrid, list_data_hybrid[6], 'X:', color='salmon', label='OpenMP Communication efficiency')

    plt.xlabel("Number of Processes")
    plt.ylabel("Efficiency (%)")
    plt.xticks(tuple(traces_procs_hybrid), tuple(label_xtics_hybrid))
    if float(max_hybrid) < 100:
        max_hybrid = 100

    plt.ylim(0, max_hybrid+5)
    plt.legend()
    plt.savefig('modelfactors_hybrid_matplot.png', bbox_inches='tight')

    ## MPI Metrics
    plt.figure()

    if count_mode == len(trace_list):
        if max(list_data[12]) != 'NaN' and max(list_data[13]) != 'NaN':
            max_mpi = max([max(list_data[9]), max(list_data[10]), max(list_data[11]),
                       max(list_data[12]), max(list_data[13])])
            plt.plot(traces_procs, list_data[9], 's-', color='green', label='MPI Parallel efficiency')
            plt.plot(traces_procs, list_data[10], 'v:', color='lime', label='MPI Load balance')
            plt.plot(traces_procs, list_data[11], 'o-.', color='lightseagreen', label='MPI Communication efficiency')
            plt.plot(traces_procs, list_data[12], linestyle=(0, (3, 10, 1, 10)), marker='s', color='gold',
                 label='Serialization efficiency')
            plt.plot(traces_procs, list_data[13], linestyle=(0, (3, 5, 1, 5)), marker='x', color='tomato',
                     label='Transfer efficiency')
        elif max(list_data[12]) == 'NaN' and max(list_data[13]) == 'NaN':
            max_mpi = max([max(list_data[9]), max(list_data[10]), max(list_data[11])])
            plt.plot(traces_procs, list_data[9], 's-', color='green', label='MPI Parallel efficiency')
            plt.plot(traces_procs, list_data[10], 'v:', color='lime', label='MPI Load balance')
            plt.plot(traces_procs, list_data[11], 'o-.', color='lightseagreen', label='MPI Communication efficiency')
        elif max(list_data[12]) != 'NaN' and max(list_data[13]) == 'NaN':
            max_mpi = max([max(list_data[9]), max(list_data[10]), max(list_data[11]),
                       max(list_data[12])])
            plt.plot(traces_procs, list_data[9], 's-', color='green', label='MPI Parallel efficiency')
            plt.plot(traces_procs, list_data[10], 'v:', color='lime', label='MPI Load balance')
            plt.plot(traces_procs, list_data[11], 'o-.', color='lightseagreen', label='MPI Communication efficiency')
            plt.plot(traces_procs, list_data[12], linestyle=(0, (3, 10, 1, 10)), marker='s', color='gold',
                     label='Serialization efficiency')
        else:
            max_mpi = max([max(list_data[9]), max(list_data[10]), max(list_data[11]),max(list_data[13])])
            plt.plot(traces_procs, list_data[9], 's-', color='green', label='MPI Parallel efficiency')
            plt.plot(traces_procs, list_data[10], 'v:', color='lime', label='MPI Load balance')
            plt.plot(traces_procs, list_data[11], 'o-.', color='lightseagreen', label='MPI Communication efficiency')
            plt.plot(traces_procs, list_data[13], linestyle=(0, (3, 5, 1, 5)), marker='x', color='tomato',
                     label='Transfer efficiency')
    else:
        if max(list_data_hybrid[4]) != 'NaN' and max(list_data_hybrid[5]) != 'NaN':
            max_mpi = max([max(list_data_hybrid[1]), max(list_data_hybrid[2]), max(list_data_hybrid[3]),
                       max(list_data_hybrid[4]), max(list_data_hybrid[5])])
            plt.plot(traces_procs_hybrid, list_data_hybrid[1], 's-', color='green', label='MPI Parallel efficiency')
            plt.plot(traces_procs_hybrid, list_data_hybrid[2], 'v:', color='lime', label='MPI Load balance')
            plt.plot(traces_procs_hybrid, list_data_hybrid[3], 'o-.', color='lightseagreen', label='MPI Communication efficiency')
            plt.plot(traces_procs_hybrid, list_data_hybrid[4], linestyle=(0, (3, 10, 1, 10)), marker='s', color='gold',
                 label='Serialization efficiency')
            plt.plot(traces_procs_hybrid, list_data_hybrid[5], linestyle=(0, (3, 5, 1, 5)), marker='x', color='tomato',
                     label='Transfer efficiency')
        elif max(list_data_hybrid[4]) == 'NaN' and max(list_data_hybrid[5]) == 'NaN':
            max_mpi = max([max(list_data_hybrid[1]), max(list_data_hybrid[2]), max(list_data_hybrid[3])])
            plt.plot(traces_procs_hybrid, list_data_hybrid[1], 's-', color='green', label='MPI Parallel efficiency')
            plt.plot(traces_procs_hybrid, list_data_hybrid[2], 'v:', color='lime', label='MPI Load balance')
            plt.plot(traces_procs_hybrid, list_data_hybrid[3], 'o-.', color='lightseagreen', label='MPI Communication efficiency')
        elif max(list_data_hybrid[4]) != 'NaN' and max(list_data_hybrid[5]) == 'NaN':
            max_mpi = max([max(list_data_hybrid[1]), max(list_data_hybrid[2]), max(list_data_hybrid[3]),
                       max(list_data_hybrid[4])])
            plt.plot(traces_procs_hybrid, list_data_hybrid[1], 's-', color='green', label='MPI Parallel efficiency')
            plt.plot(traces_procs_hybrid, list_data_hybrid[2], 'v:', color='lime', label='MPI Load balance')
            plt.plot(traces_procs_hybrid, list_data_hybrid[3], 'o-.', color='lightseagreen', label='MPI Communication efficiency')
            plt.plot(traces_procs_hybrid, list_data_hybrid[4], linestyle=(0, (3, 10, 1, 10)), marker='s', color='gold',
                     label='Serialization efficiency')
        else:
            max_mpi = max([max(list_data_hybrid[1]), max(list_data_hybrid[2]), max(list_data_hybrid[3]),
                           max(list_data_hybrid[5])])
            plt.plot(traces_procs_hybrid, list_data_hybrid[1], 's-', color='green', label='MPI Parallel efficiency')
            plt.plot(traces_procs_hybrid, list_data_hybrid[2], 'v:', color='lime', label='MPI Load balance')
            plt.plot(traces_procs_hybrid, list_data_hybrid[3], 'o-.', color='lightseagreen', label='MPI Communication efficiency')
            plt.plot(traces_procs_hybrid, list_data_hybrid[5], linestyle=(0, (3, 5, 1, 5)), marker='x', color='tomato',
                     label='Transfer efficiency')

    plt.xlabel("Number of Processes")
    plt.ylabel("Efficiency (%)")
    plt.xticks(tuple(traces_procs_hybrid), tuple(label_xtics_hybrid))
    if float(max_mpi) < 100:
        max_mpi = 100

    plt.ylim(0, max_mpi+5)
    plt.legend()
    plt.savefig('modelfactors_mpi_matplot.png', bbox_inches='tight')


def plots_speedup_matplot(trace_list, trace_processes, trace_tasks, trace_threads, trace_mode, cmdl_args):
    # Plotting using python
    # For plotting using python, read the csv file
    file_path = os.path.join(os.getcwd(), 'other_metrics.csv')
    df = pd.read_csv(file_path, sep=';')

    traces_procs = list(df.keys())[1:]

    list_data = []
    for index, rows in df.iterrows():
        list_temp = []
        for value in list(rows)[1:]:
            if value != 'Non-Avail':
                list_temp.append(float(value))
            elif value == 'Non-Avail':
                list_temp.append('NaN')
        # print(list_temp)
        list_data.append(list_temp)

    # To control same number of processes for the header on plots and table
    same_procs = True
    procs_trace_prev = trace_processes[trace_list[0]]
    tasks_trace_prev = trace_tasks[trace_list[0]]
    threads_trace_prev = trace_threads[trace_list[0]]
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        if procs_trace_prev == trace_processes[trace] and tasks_trace_prev == tasks \
                and threads_trace_prev == threads:
            same_procs *= True
        else:
            same_procs *= False

    # Set limit for projection
    if cmdl_args.limit:
        limit = cmdl_args.limit
    else:
        limit = str(trace_processes[trace_list[len(trace_list) - 1]])

    limit_min = trace_processes[trace_list[0]]

    # proc_ratio to ideal speedup
    proc_ratio = []
    for index, trace in enumerate(trace_list):
        proc_ratio.append(trace_processes[trace]/trace_processes[trace_list[0]])
    # print(proc_ratio)

    # To xticks label
    label_xtics = []
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        if int(limit) == int(limit_min) and same_procs:
            s_xtics = str(trace_processes[trace]) + '[' + str(index + 1) + ']'
        elif int(limit) == int(limit_min) and not same_procs:
            if trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
                s_xtics = str(trace_processes[trace]) + '(' + str(tasks) + 'x' \
                          + str(threads) + ')'
        else:
            s_xtics = str(trace_processes[trace])
        s_xtics += '[' + str(index + 1) + ']'
        label_xtics.append(s_xtics)

    int_traces_procs = []
    prev_procs = int(float(traces_procs[0]))
    int_traces_procs.append(int(float(traces_procs[0])))
    count_rep = 0
    for procs in traces_procs[1:]:
        if prev_procs == int(float(procs)):
            count_rep += 1
            int_traces_procs.append(int(float(procs)) + (6*count_rep))
            prev_procs = int(float(procs))
        else:
            int_traces_procs.append(int(float(procs)))
            prev_procs = int(float(procs))
            count_rep = 0

    ### Plot: Global Metrics
    plt.figure()
    for x, y in zip(int_traces_procs, list_data[2]):
        label = "{:.2f}".format(y)
        plt.annotate(label, (x, y), textcoords="offset points", xytext=(0, 10), ha='center')

    plt.plot(int_traces_procs, list_data[2], 'o-', color='blue', label='measured')
    plt.plot(int_traces_procs, proc_ratio, 'o-', color='black', label='ideal')
    plt.xlabel("Number of Processes")
    plt.ylabel("SpeedUp")
    plt.xticks(tuple(int_traces_procs), tuple(label_xtics))
    #plt.yscale('log')
    plt.legend()
    #plt.xlim(0, )
    plt.ylim(0, )
    plt.savefig('speedup_matplot.png', bbox_inches='tight')

    ### Plot: Efficiency
    # print(list_data)
    plt.figure()
    for x, y in zip(int_traces_procs, list_data[1]):
        label = "{:.2f}".format(y)
        plt.annotate(label, (x, y), textcoords="offset points", xytext=(0, 10), ha='center')

    plt.plot(int_traces_procs, list_data[1], 'o-', color='blue', label='measured')
    plt.axhline(y=1, color='black', linestyle='-', label='ideal')
    plt.xlabel("Number of Processes")
    plt.ylabel("Efficiency")
    plt.xticks(tuple(int_traces_procs), tuple(label_xtics))
    # plt.yscale('log')
    max_y = max(list_data[1])
    if max_y < 1.1:
        max_y = 1.0
    #plt.xlim(0, )
    plt.ylim(0,max_y+0.1)
    plt.legend()
    plt.savefig('efficiency_matplot.png', bbox_inches='tight')

def print_talp_metrics_csv(device_factors,host_factors, trace_list, trace_processes,raw_data):
    """Prints the model factors table in a csv file."""
    global mod_device_factors_doc, mod_host_factors_doc

    delimiter = ','
    # File is stored in the trace directory
    # file_path = os.path.join(os.path.dirname(os.path.realpath(trace_list[0])), 'modelfactors.csv')
    # File is stored in the execution directory
    file_path = os.path.join(os.getcwd(), 'talp_metrics.csv')

    with open(file_path, 'w') as output:       
        line = "\"#Proc(#GPU)\""
        for trace in trace_list:
            line += delimiter
            line += str(trace_processes[trace]) + "("+ str(raw_data['count_devices'][trace])+")"
        output.write(line + '\n')

        for mod_key in mod_host_factors_doc:
            line = "\"" + mod_host_factors_doc[mod_key].replace('  ', '', 2)+ "\""
            for trace in trace_list:
                line += delimiter
                try:  # except NaN
                    if host_factors[mod_key][trace] == "Non-Avail":
                        line += '0.00'
                    else:
                        line += '{0:.6f}'.format(host_factors[mod_key][trace])
                except ValueError:
                    line += '{}'.format(host_factors[mod_key][trace])
            output.write(line + '\n')

        for mod_key in mod_device_factors_doc:
            line = "\"" + mod_device_factors_doc[mod_key].replace('  ', '', 2)+ "\""
            for trace in trace_list:
                line += delimiter
                try:  # except NaN
                    if device_factors[mod_key][trace] == "Non-Avail":
                        line += '0.00'
                    else:
                        line += '{0:.6f}'.format(device_factors[mod_key][trace])
                except ValueError:
                    line += '{}'.format(device_factors[mod_key][trace])
            output.write(line + '\n')
        

        #output.write('#\n')

    print('')
    print('======== Output File: Device Metrics ========')
    print('Device Metrics written to ' + file_path)


def plots_talp_efficiency_table_matplot(trace_list, trace_processes, trace_tasks, trace_threads, trace_mode,raw_data, cmdl_args):
    # Plotting using python
    # For plotting using python, read the csv file

    file_path = os.path.join(os.getcwd(), 'talp_metrics.csv')
    df = pd.read_csv(file_path)
    metrics = df['#Proc(#GPU)'].tolist()

    ## remove complete nan columns and put to DF1
    df1 = df.dropna(axis='columns', how='all')

    traces_procs = list(df.keys())[1:]
    # To control same number of processes for the header on plots and table
    same_procs = True
    procs_trace_prev = trace_processes[trace_list[0]]
    tasks_trace_prev = trace_tasks[trace_list[0]]
    threads_trace_prev = trace_threads[trace_list[0]]
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        if procs_trace_prev == trace_processes[trace] and tasks_trace_prev == tasks \
                and threads_trace_prev == threads:
            same_procs *= True
        else:
            same_procs *= False

    # Set limit for projection
    if cmdl_args.limit:
        limit = cmdl_args.limit
    else:
        limit = str(trace_processes[trace_list[len(trace_list)-1]])

    limit_min = trace_processes[trace_list[0]]

    # To xticks label
    label_xtics = []
    label_xtics_hybrid = []
    for index, trace in enumerate(trace_list):
        tasks = trace_tasks[trace]
        threads = trace_threads[trace]
        #if int(limit) == int(limit_min) and same_procs:
        #    s_xtics = str(trace_processes[trace]) + '[' + str(index + 1) + ']'
        #else:
        if trace_mode[trace] == "Detailed+MPI+CUDA":
            s_xtics = str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')' \
                          +"("+ str(raw_data['count_devices'][trace])+"GPUs)"+ '[' + str(index + 1) + ']'
            label_xtics_hybrid.append(s_xtics)
        elif trace_mode[trace][0:len("Detailed+MPI+")] == "Detailed+MPI+":
            s_xtics = str(trace_processes[trace]) + '(' + str(tasks) + 'x' + str(threads) + ')' \
                          + '[' + str(index + 1) + ']'
            label_xtics_hybrid.append(s_xtics)
        else:
            s_xtics = str(trace_processes[trace]) + '[' + str(index + 1) + ']'
        label_xtics.append(s_xtics)
    ##### End xticks

    # BEGIN To adjust header to big number of processes
    max_len_header = 7
    for labelx in label_xtics:
        if len(labelx) > max_len_header:
            max_len_header = len(labelx)

    # END To adjust header to big number of processes

    list_data = []

    for index, rows in df1.iterrows():
        list_temp = []
        for value in list(rows)[1:]:
            if pd.isna(value) or value == 0.0:
                list_temp.append(np.nan)
            else:
                list_temp.append(float(value))
        list_data.append(list_temp)

    list_np = np.array(list_data)

    idx = metrics
    #cols = label_xtics
    cols = label_xtics_hybrid
    # cols = traces_procs
    df = pd.DataFrame(list_np, index=idx, columns=cols)

    # min for 1 trace is x=3 for the (x,y) in figsize
    size_figure_y = len(idx) * 0.40
    size_figure_x = len(cols) * 0.14 * max_len_header
    plt.figure(figsize=(size_figure_x, size_figure_y))

    ax = sns.heatmap(df, cmap='RdYlGn', linewidths=0.05, annot=True, vmin=0, vmax=100, center=75, \
                     fmt='.2f', annot_kws={"size": 10}, cbar_kws={'label': 'Percentage(%)'})
    ## to align ylabels to left
    plt.yticks(rotation=0, ha='left')

    ax.xaxis.tick_top()
    # to adjust metrics
    len_pad = 0
    for metric in metrics:
        if len(metric) > len_pad:
            len_pad = len(metric)

    ax.yaxis.set_tick_params(pad=len_pad + 164)

    plt.savefig('efficiency_table_talp_matplot.png', bbox_inches='tight')

    
