/*****************************************************************************\
 *                        ANALYSIS PERFORMANCE TOOLS                         *
 *                                   Extrae                                  *
 *              Instrumentation package for parallel applications            *
 *****************************************************************************
 *     ___     This library is free software; you can redistribute it and/or *
 *    /  __         modify it under the terms of the GNU LGPL as published   *
 *   /  /  _____    by the Free Software Foundation; either version 2.1      *
 *  /  /  /     \   of the License, or (at your option) any later version.   *
 * (  (  ( B S C )                                                           *
 *  \  \  \_____/   This library is distributed in hope that it will be      *
 *   \  \__         useful but WITHOUT ANY WARRANTY; without even the        *
 *    \___          implied warranty of MERCHANTABILITY or FITNESS FOR A     *
 *                  PARTICULAR PURPOSE. See the GNU LGPL for more details.   *
 *                                                                           *
 * You should have received a copy of the GNU Lesser General Public License  *
 * along with this library; if not, write to the Free Software Foundation,   *
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA          *
 * The GNU LEsser General Public License is contained in the file COPYING.   *
 *                                 ---------                                 *
 *   Barcelona Supercomputing Center - Centro Nacional de Supercomputacion   *
\*****************************************************************************/

#include "common.h"

#ifdef HAVE_DLFCN_H
# define __USE_GNU
# include <dlfcn.h>
# undef  __USE_GNU
#endif
#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#endif
#ifdef HAVE_STDARG_H
# include <stdarg.h>
#endif

#include <omp.h>
#include "omp_common.h"
#include "gnu_libgomp.h"
#include "gomp_helpers.h"
#include "nested_id.h"
#include "xalloc.h"


/* We detect the version of OpenMP specification implemented by the compiler searching for 
 * key API functions known to be available since specific versions (see xtr_GOMP_load_hooks)
 */
float __GOMP_version = 0;

/*
 * Relation of GNU compiler versions and OpenMP specification they implement, 
 * from: https://gcc.gnu.org/projects/gomp
 *       https://gcc.gnu.org/wiki/openmp
 *       https://en.wikipedia.org/wiki/OpenMP#Implementations
 * 
 * +----------------------+--------------------------------------------------------------------------------------------+
 * | GNU compiler version |                                       OpenMP specification                                 |
 * +----------------------+--------------------------------------------------------------------------------------------+
 * |                  4.2 | 2.5                                                                                        |
 * |                  4.4 | 3.0                                                                                        |
 * |                  4.7 | 3.1                                                                                        |
 * |                  4.9 | 4.0 (C/C++)                                                                                |
 * |                4.9.1 | 4.0 (Fortran)                                                                              |
 * |                  5.0 | Offloading support                                                                         |
 * |                  6.0 | 4.5 (C/C++)                                                                                |
 * |                  7.0 | 4.5 (Fortran initial support)                                                              |
 * |                  9.0 | 5.0 (C/C++ initial support)                                                                |
 * |                 10.0 | 5.0 (C/C++ more features; Fortran initial support)                                         |
 * |                 11.0 | 5.0 (C/C++); 4.5 (Fortran full support); 5.0 (Fortran more features); Nonrectangular loops |
 * |                 12.0 | 5.1                                                                                        |
 * |                 13.0 | 5.2                                                                                        |
 * +----------------------+--------------------------------------------------------------------------------------------+
 *
 * Check this file from libgomp to see all functions available in the public API for a given compiler version:
 * https://github.com/gcc-mirror/gcc/blob/releases/gcc-11.1.0/libgomp/libgomp_g.h
*/

/*********************************/
/***** Available since GCC 4.2 ***/
/*********************************/

/* libgomp/barrier.c */

void (*REAL_SYMBOL_PTR(GOMP_barrier)) (void) = NULL;

/* libgomp/critical.c */

void (*REAL_SYMBOL_PTR(GOMP_critical_start)) (void) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_critical_end)) (void) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_critical_name_start)) (void **) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_critical_name_end)) (void **) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_atomic_start)) (void) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_atomic_end)) (void) = NULL;

/* libgomp/icv.c*/

void (*REAL_SYMBOL_PTR(omp_set_num_threads)) (int) = NULL;

/* Fortran-mangled function */

void (*REAL_SYMBOL_PTR(omp_set_num_threads_)) (int *) = NULL;

/* libgomp/lock.c */

void (*REAL_SYMBOL_PTR(omp_set_lock)) (omp_lock_t *) = NULL;
void (*REAL_SYMBOL_PTR(omp_unset_lock)) (omp_lock_t *) = NULL;

/* Fortran-mangled functions */

void (*REAL_SYMBOL_PTR(omp_set_lock_)) (omp_lock_t *) = NULL;
void (*REAL_SYMBOL_PTR(omp_unset_lock_)) (omp_lock_t *) = NULL;

/* libgomp/loop.c */

int (*REAL_SYMBOL_PTR(GOMP_loop_static_start)) (long, long, long, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_dynamic_start)) (long, long, long, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_guided_start)) (long, long, long, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_runtime_start)) (long, long, long, long, long *, long *) = NULL;

int (*REAL_SYMBOL_PTR(GOMP_loop_ordered_static_start)) (long, long, long, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_ordered_dynamic_start)) (long, long, long, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_ordered_guided_start)) (long, long, long, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_ordered_runtime_start)) (long, long, long, long, long *, long *) = NULL;

int (*REAL_SYMBOL_PTR(GOMP_loop_static_next)) (long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_dynamic_next)) (long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_guided_next)) (long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_runtime_next)) (long *, long *) = NULL;

int (*REAL_SYMBOL_PTR(GOMP_loop_ordered_static_next)) (long *, long *) = NULL; 
int (*REAL_SYMBOL_PTR(GOMP_loop_ordered_dynamic_next)) (long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_ordered_guided_next)) (long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_ordered_runtime_next)) (long *, long *) = NULL;

void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_static_start)) (void*, void*, unsigned, long, long, long, long) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_dynamic_start)) (void*, void*, unsigned, long, long, long, long) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_guided_start)) (void*, void*, unsigned, long, long, long, long) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_runtime_start)) (void*, void*, unsigned, long, long, long) = NULL;

void (*REAL_SYMBOL_PTR(GOMP_loop_end)) (void) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_loop_end_nowait)) (void) = NULL;

/* libgomp/ordered.c */

void (*REAL_SYMBOL_PTR(GOMP_ordered_start)) (void) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_ordered_end)) (void) = NULL;

/* libgomp/parallel.c */

void (*REAL_SYMBOL_PTR(GOMP_parallel_start)) (void *, void *, unsigned) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_parallel_end)) (void) = NULL;

/* libgomp/sections.c */

unsigned (*REAL_SYMBOL_PTR(GOMP_sections_start)) (unsigned) = NULL;
unsigned (*REAL_SYMBOL_PTR(GOMP_sections_next)) (void) = NULL;

void (*REAL_SYMBOL_PTR(GOMP_parallel_sections_start)) (void *, void *, unsigned, unsigned) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_sections_end)) (void) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_sections_end_nowait)) (void) = NULL;

/* libgomp/single.c */

unsigned (*REAL_SYMBOL_PTR(GOMP_single_start)) (void) = NULL;


/*********************************/
/***** Available since GCC 4.4 ***/
/*********************************/

/* libgomp/team.c, libgomp/task.c */

// GOMP_task appeared in GCC 4.4 but increased the parameters in 4.9 and later in 6.0 and 9.0
void (*REAL_SYMBOL_PTR(GOMP_task)) (void *, void *, void *, long, long, int, unsigned, ...) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_taskwait)) (void) = NULL;


/*********************************/
/***** Available since GCC 4.7 ***/
/*********************************/

/* libgomp/task.c */
void (*REAL_SYMBOL_PTR(GOMP_taskyield)) (void) = NULL;

/*********************************/
/***** Available since GCC 4.9 ***/
/*********************************/

/* libgomp/loop.c */

void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_static)) (void*, void*, unsigned, long, long, long, long, unsigned) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_dynamic)) (void*, void*, unsigned, long, long, long, long, unsigned) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_guided)) (void*, void*, unsigned, long, long, long, long, unsigned) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_runtime)) (void*, void*, unsigned, long, long, long, unsigned) = NULL;

/* libgomp/task.c */

void (*REAL_SYMBOL_PTR(GOMP_taskgroup_start)) (void) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_taskgroup_end)) (void) = NULL;

/* libgomp/parallel.c */

void (*REAL_SYMBOL_PTR(GOMP_parallel)) (void *, void *, unsigned, unsigned) = NULL;

/* libgomp/sections.c */

void (*REAL_SYMBOL_PTR(GOMP_parallel_sections)) (void *, void *, unsigned, unsigned, unsigned) = NULL;

/* libgomp/target.c */

void (*REAL_SYMBOL_PTR(GOMP_target)) (int, void *, void *, size_t, void **, size_t *, unsigned char *) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_target_data)) (int, void *, size_t, void **, size_t *, unsigned char *) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_target_end_data)) (void) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_target_update)) (int, void *, size_t, void **, size_t *, unsigned char *) = NULL;


/*********************************/
/***** Available since GCC 6.0 ***/
/*********************************/

/* libgomp/loop.c */

int (*REAL_SYMBOL_PTR(GOMP_loop_nonmonotonic_dynamic_start)) (long, long, long, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_nonmonotonic_guided_start)) (long, long, long, long, long *, long *) = NULL;

int (*REAL_SYMBOL_PTR(GOMP_loop_nonmonotonic_dynamic_next)) (long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_nonmonotonic_guided_next)) (long *, long *) = NULL;

int (*REAL_SYMBOL_PTR(GOMP_loop_doacross_static_start)) (unsigned, long *, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_doacross_dynamic_start)) (unsigned, long *, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_doacross_guided_start)) (unsigned, long *, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_doacross_runtime_start)) (unsigned, long *, long *, long *) = NULL;

void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_nonmonotonic_dynamic)) (void *, void *, unsigned, long, long, long, long, unsigned) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_nonmonotonic_guided)) (void *, void *, unsigned, long, long, long, long, unsigned) = NULL;

/* libgomp/ordered.c */

void (*REAL_SYMBOL_PTR(GOMP_doacross_post)) (long *) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_doacross_wait)) (long, ...) = NULL;

/* libgomp/taskloop.c */

void (*REAL_SYMBOL_PTR(GOMP_taskloop)) (void *, void *, void *, long, long, unsigned, unsigned long, int, long, long, long) = NULL;


/*********************************/
/***** Available since GCC 6.1 ***/
/*********************************/

/* libgomp/target.c */

void (*REAL_SYMBOL_PTR(GOMP_target_ext)) (int, void *, size_t, void **, size_t *, unsigned short *, unsigned int, void **, void **) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_target_data_ext)) (int, size_t, void **, size_t *, unsigned short *) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_target_update_ext)) (int, size_t, void **, size_t *, unsigned short *, unsigned int, void **) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_target_enter_exit_data)) (int, size_t, void **, size_t *, unsigned short *, unsigned int, void **) = NULL;


/*********************************/
/***** Available since GCC 9.0 ***/
/*********************************/

/* libgomp/loop.c */

int (*REAL_SYMBOL_PTR(GOMP_loop_nonmonotonic_runtime_start)) (long, long, long, long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_maybe_nonmonotonic_runtime_start)) (long, long, long, long *, long *) = NULL;

int (*REAL_SYMBOL_PTR(GOMP_loop_nonmonotonic_runtime_next)) (long *, long *) = NULL;
int (*REAL_SYMBOL_PTR(GOMP_loop_maybe_nonmonotonic_runtime_next)) (long *, long *) = NULL;

void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_nonmonotonic_runtime)) (void *, void *, unsigned, long, long, long, unsigned) = NULL;
void (*REAL_SYMBOL_PTR(GOMP_parallel_loop_maybe_nonmonotonic_runtime)) (void *, void *, unsigned, long, long, long, unsigned) = NULL;

/* libgomp/taskloop.c */

void (*REAL_SYMBOL_PTR(GOMP_taskloop_ull)) (void *, void *, void *, long, long, unsigned, unsigned long, int, unsigned long long, unsigned long long, unsigned long long) = NULL;

/* libgomp/teams.c */

void (*REAL_SYMBOL_PTR(GOMP_teams_reg)) (void *, void *, unsigned int, unsigned int, unsigned int) = NULL;

/**
 * xtr_GOMP_extra_debug
 * 
 * Fills buffer with formatted debug information regarding the current nesting level and ancestor ids for the calling thread
 */
void xtr_GOMP_extra_debug(char *buffer, int buffer_size)
{
	xtr_nested_id_t tid;

	char *tid_str = NULL;
	XTR_NESTED_ID_NEW(tid, omp_get_level, omp_get_ancestor_thread_num);
	tid_str = xtr_nested_id_tostr(&tid);

	snprintf(buffer, buffer_size, "LVL:%d%s%s", tid.level,
						(tid.level > 1 ? " TID:" : ""),
						(tid.level > 1 ? tid_str : ""));

	XTR_NESTED_ID_FREE(tid);
	xfree(tid_str);
}

/******************************************************************************\
 *                                                                            *
 *                             INITIALIZATIONS                                *
 *                                                                            *
\******************************************************************************/

/**
 * xtr_GOMP_load_hooks
 *
 * Find the real implementation of the functions. We use dlsym to find the next
 * definition of the different symbols of the OpenMP runtime (i.e. skip our
 * wrapper, find the real one). 
 *
 * @return 1 if pointers to all real symbols are found; 0 otherwise
 */
static int xtr_GOMP_load_hooks (void)
{
	int nhooks = 0;
	char *env_GOMP_version = NULL;

	// Detect the OpenMP version supported by the runtime
	if ((env_GOMP_version = getenv("EXTRAE_GOMP_VERSION")) != NULL) 
	{
		__GOMP_version = strtof(env_GOMP_version, NULL);

		if ((__GOMP_version != GOMP_API_5_2) &&
				(__GOMP_version != GOMP_API_5_1) &&
				(__GOMP_version != GOMP_API_5_0) &&
				(__GOMP_version != GOMP_API_4_5) &&
				(__GOMP_version != GOMP_API_4_0) &&
				(__GOMP_version != GOMP_API_3_1)) {
			fprintf(stderr, PACKAGE_NAME": ERROR! Unsupported GOMP version (%.1f). Valid versions are: %.1f, %.1f, %.1f, %.1f, %.1f and %.1f. Exiting...\n",
				__GOMP_version, GOMP_API_3_1, GOMP_API_4_0, GOMP_API_4_5, GOMP_API_5_0, GOMP_API_5_1, GOMP_API_5_2);
			exit (-1);
		}
	} else if (dlsym(RTLD_NEXT, "omp_in_explicit_task") != NULL) { // XXX Find a GOMP routine to identify runtime version, just in case a modern KMP runtime has legacy GNU support 
		__GOMP_version = GOMP_API_5_2;
	} else if (dlsym(RTLD_NEXT, "GOMP_teams4") != NULL) {
		__GOMP_version = GOMP_API_5_1;
	} else if (dlsym(RTLD_NEXT, "GOMP_teams_reg") != NULL) {
		__GOMP_version = GOMP_API_5_0;
	} else if (dlsym(RTLD_NEXT, "GOMP_taskloop") != NULL) {
		__GOMP_version = GOMP_API_4_5;
	} else if (dlsym(RTLD_NEXT, "GOMP_taskgroup_start") != NULL) {
		__GOMP_version = GOMP_API_4_0;
	} else {
		__GOMP_version = GOMP_API_3_1;
	}

	MASTER_OUT("Detected GOMP version is %.1f\n", __GOMP_version);

	// Initialize pointers to the real GOMP symbols (check libgomp/libgomp_g.h) for the full list 
	
	/* libgomp/barrier.c */

	OMP_HOOK_INIT(GOMP_barrier, nhooks);

	/* libgomp/critical.c */

	OMP_HOOK_INIT(GOMP_critical_start, nhooks);
	OMP_HOOK_INIT(GOMP_critical_end, nhooks);
	OMP_HOOK_INIT(GOMP_critical_name_start, nhooks);
	OMP_HOOK_INIT(GOMP_critical_name_end, nhooks);
	OMP_HOOK_INIT(GOMP_atomic_start, nhooks);
	OMP_HOOK_INIT(GOMP_atomic_end, nhooks);

	/* libgomp/loop.c */

	OMP_HOOK_INIT(GOMP_loop_static_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_dynamic_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_guided_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_runtime_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_nonmonotonic_dynamic_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_nonmonotonic_guided_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_nonmonotonic_runtime_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_maybe_nonmonotonic_runtime_start, nhooks);

	OMP_HOOK_INIT(GOMP_loop_ordered_static_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_ordered_dynamic_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_ordered_guided_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_ordered_runtime_start, nhooks);

	OMP_HOOK_INIT(GOMP_loop_static_next, nhooks);
	OMP_HOOK_INIT(GOMP_loop_dynamic_next, nhooks);
	OMP_HOOK_INIT(GOMP_loop_guided_next, nhooks);
	OMP_HOOK_INIT(GOMP_loop_runtime_next, nhooks);
	OMP_HOOK_INIT(GOMP_loop_nonmonotonic_dynamic_next, nhooks);
	OMP_HOOK_INIT(GOMP_loop_nonmonotonic_guided_next, nhooks);
	OMP_HOOK_INIT(GOMP_loop_nonmonotonic_runtime_next, nhooks);
	OMP_HOOK_INIT(GOMP_loop_maybe_nonmonotonic_runtime_next, nhooks);

	OMP_HOOK_INIT(GOMP_loop_ordered_static_next, nhooks);
	OMP_HOOK_INIT(GOMP_loop_ordered_dynamic_next, nhooks);
	OMP_HOOK_INIT(GOMP_loop_ordered_guided_next, nhooks);
	OMP_HOOK_INIT(GOMP_loop_ordered_runtime_next, nhooks);

	OMP_HOOK_INIT(GOMP_loop_doacross_static_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_doacross_dynamic_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_doacross_guided_start, nhooks);
	OMP_HOOK_INIT(GOMP_loop_doacross_runtime_start, nhooks);

	OMP_HOOK_INIT(GOMP_parallel_loop_static_start, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_dynamic_start, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_guided_start, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_runtime_start, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_static, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_dynamic, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_guided, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_runtime, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_nonmonotonic_dynamic, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_nonmonotonic_guided, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_nonmonotonic_runtime, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_loop_maybe_nonmonotonic_runtime, nhooks);

	OMP_HOOK_INIT(GOMP_loop_end, nhooks);
	OMP_HOOK_INIT(GOMP_loop_end_nowait, nhooks);

	/* libgomp/ordered.c */

	OMP_HOOK_INIT(GOMP_ordered_start, nhooks);
	OMP_HOOK_INIT(GOMP_ordered_end, nhooks);
	OMP_HOOK_INIT(GOMP_doacross_post, nhooks);
	OMP_HOOK_INIT(GOMP_doacross_wait, nhooks);

	/* libgomp/parallel.c */

	OMP_HOOK_INIT(GOMP_parallel_start, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_end, nhooks);
	OMP_HOOK_INIT(GOMP_parallel, nhooks);

	/* libgomp/task.c */

	OMP_HOOK_INIT(GOMP_task, nhooks);
	OMP_HOOK_INIT(GOMP_taskloop, nhooks);
	OMP_HOOK_INIT(GOMP_taskloop_ull, nhooks);
	OMP_HOOK_INIT(GOMP_taskwait, nhooks);
	OMP_HOOK_INIT(GOMP_taskyield, nhooks);
	OMP_HOOK_INIT(GOMP_taskgroup_start, nhooks);
	OMP_HOOK_INIT(GOMP_taskgroup_end, nhooks);

	/* libgomp/sections.c */

	OMP_HOOK_INIT(GOMP_sections_start, nhooks);
	OMP_HOOK_INIT(GOMP_sections_next, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_sections_start, nhooks);
	OMP_HOOK_INIT(GOMP_parallel_sections, nhooks);
	OMP_HOOK_INIT(GOMP_sections_end, nhooks);
	OMP_HOOK_INIT(GOMP_sections_end_nowait, nhooks);

	/* libgomp/single.c */

	OMP_HOOK_INIT(GOMP_single_start, nhooks);

	/* libgomp/teams.c */

	OMP_HOOK_INIT(GOMP_teams_reg, nhooks);

	/* libgomp/target.c */

	OMP_HOOK_INIT(GOMP_target, nhooks);
	OMP_HOOK_INIT(GOMP_target_data, nhooks);
	OMP_HOOK_INIT(GOMP_target_end_data, nhooks);
	OMP_HOOK_INIT(GOMP_target_update, nhooks);
	OMP_HOOK_INIT(GOMP_target_enter_exit_data, nhooks);
	OMP_HOOK_INIT(GOMP_target_ext, nhooks);
	OMP_HOOK_INIT(GOMP_target_data_ext, nhooks);
	OMP_HOOK_INIT(GOMP_target_update_ext, nhooks);

	if (nhooks <= 0) 
	{
		MASTER_WARN("OpenMP GOMP module was activated but no symbols were hooked during startup. Switching to deferred in-wrapper initialization.\n");
	}
	else
	{
		MASTER_OUT("Successfully loaded %d hooks from libGOMP during initialization\n", nhooks);
	}

	return (nhooks > 0);
}

/**
 * xtr_OMP_GOMP_init
 *
 * Initializes the instrumentation module for GNU libgomp.
 *
 * @param rank The current process ID 
 */
int xtr_OMP_GOMP_init (void)
{
	__GOMP_init_helpers();

	return xtr_GOMP_load_hooks();
}

