/*****************************************************************************\
 *                        ANALYSIS PERFORMANCE TOOLS                         *
 *                               libparaver-api                              *
 *                      API Library for libparaver-kernel                    *
 *****************************************************************************
 *     ___     This library is free software; you can redistribute it and/or *
 *    /  __         modify it under the terms of the GNU LGPL as published   *
 *   /  /  _____    by the Free Software Foundation; either version 2.1      *
 *  /  /  /     \   of the License, or (at your option) any later version.   *
 * (  (  ( B S C )                                                           *
 *  \  \  \_____/   This library is distributed in hope that it will be      *
 *   \  \__         useful but WITHOUT ANY WARRANTY; without even the        *
 *    \___          implied warranty of MERCHANTABILITY or FITNESS FOR A     *
 *                  PARTICULAR PURPOSE. See the GNU LGPL for more details.   *
 *                                                                           *
 * You should have received a copy of the GNU Lesser General Public License  *
 * along with this library; if not, write to the Free Software Foundation,   *
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA          *
 * The GNU LEsser General Public License is contained in the file COPYING.   *
 *                                 ---------                                 *
 *   Barcelona Supercomputing Center - Centro Nacional de Supercomputacion   *
\*****************************************************************************/


#include <map>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "eventlabels.h"
#include "kernelconnection.h"
#include "tracecutter.h"
#include "traceoptions.h"

using namespace std;

string TraceCutter::traceToolID = "cutter";
string TraceCutter::traceToolName = "Cutter";
string TraceCutter::traceToolExtension = "chop";

TraceCutter *TraceCutter::create( const KernelConnection *whichKernel,
                                  string traceIn,
                                  string traceOut,
                                  TraceOptions *options,
                                  ProgressController *progress )
{
  return new TraceCutterProxy( whichKernel, traceIn, traceOut, options, progress );
}


string TraceCutter::getID()
{
  return TraceCutter::traceToolID;
}


string TraceCutter::getName()
{
  return TraceCutter::traceToolName;
}


string TraceCutter::getExtension()
{
  return TraceCutter::traceToolExtension;
}

TraceCutterProxy::TraceCutterProxy( const KernelConnection *whichKernel,
                                    string traceIn,
                                    string traceOut,
                                    TraceOptions *options,
                                    ProgressController *progress )
{
  string pcf_name;
  FILE *pcfFile;
#ifdef _WIN32
  struct _stat tmpStatBuffer;
#else
  struct stat tmpStatBuffer;
#endif

  pcf_name = LocalKernel::composeName( traceIn, string( "pcf" ) );
  int statReturn;
#ifdef _WIN32
  statReturn = _stat( pcf_name.c_str(), &tmpStatBuffer );
#else
  statReturn = stat( pcf_name.c_str(), &tmpStatBuffer );
#endif

  vector< TEventType > HWCTypes;
  vector< TEventType > notHWCTypes;
  if( statReturn == 0 && tmpStatBuffer.st_size > 0 )
  {
    PCFFileParser<> pcfParser( pcf_name );

    EventLabels myEventLabels = EventLabels( pcfParser );
    vector< TEventType > allTypes;
    myEventLabels.getTypes( allTypes );
    for( vector< TEventType >::iterator it = allTypes.begin(); it != allTypes.end(); ++it )
    {
      if( *it >= 42000000 && *it < 50000000 )
        HWCTypes.push_back( *it );
      else
      {
        auto tmpValues = pcfParser.getEventValues( *it );
        if( tmpValues.empty() )
          notHWCTypes.push_back( *it );
      }
    }
  }

  myTraceCutter = whichKernel->newTraceCutter( options, HWCTypes, notHWCTypes );
}


TraceCutterProxy::~TraceCutterProxy()
{
  delete myTraceCutter;
}


void TraceCutterProxy::execute( string trace_in, string trace_out, ProgressController *progress )
{
  myTraceCutter->execute( trace_in, trace_out, progress );
}


void TraceCutterProxy::set_by_time( bool whichByTime )
{
  myTraceCutter->set_by_time( whichByTime );
}


void TraceCutterProxy::set_min_cutting_time( unsigned long long minCutTime )
{
  myTraceCutter->set_min_cutting_time( minCutTime );
}

void TraceCutterProxy::set_max_cutting_time( unsigned long long maxCutTime )
{
  myTraceCutter->set_min_cutting_time( maxCutTime );
}

void TraceCutterProxy::set_minimum_time_percentage( unsigned long long minimumPercentage )
{
  myTraceCutter->set_minimum_time_percentage( minimumPercentage );
}

void TraceCutterProxy::set_maximum_time_percentage( unsigned long long maximumPercentage )
{
  myTraceCutter->set_maximum_time_percentage( maximumPercentage );
}

void TraceCutterProxy::set_tasks_list( char *tasksList )
{
  myTraceCutter->set_tasks_list( tasksList );
}

void TraceCutterProxy::set_original_time( bool originalTime )
{
  myTraceCutter->set_original_time ( originalTime );
}

void TraceCutterProxy::set_max_trace_size( int traceSize )
{
  myTraceCutter->set_max_trace_size( traceSize );
}

void TraceCutterProxy::set_break_states( bool breakStates )
{
  myTraceCutter->set_break_states( breakStates );
}

void TraceCutterProxy::set_remFirstStates( bool remStates )
{
  myTraceCutter->set_remFirstStates ( remStates );
}

void TraceCutterProxy::set_remLastStates( bool remStates )
{
  myTraceCutter->set_remLastStates( remStates );
}

void TraceCutterProxy::set_keep_boundary_events( bool keepBoundaryEvents )
{
  myTraceCutter->set_keep_boundary_events( keepBoundaryEvents );
}

void TraceCutterProxy::set_keep_all_events( bool keepAllEvents )
{
  myTraceCutter->set_keep_all_events( keepAllEvents );
}

void TraceCutterProxy::setCutterApplicationCaller( string caller )
{
  myTraceCutter->setCutterApplicationCaller( caller );
}


