/*****************************************************************************\
 *                        ANALYSIS PERFORMANCE TOOLS                         *
 *                            ptools_common_files                            *
 *                   Performance tools common files library                  *
 *****************************************************************************
 *     ___     This library is free software; you can redistribute it and/or *
 *    /  __         modify it under the terms of the GNU LGPL as published   *
 *   /  /  _____    by the Free Software Foundation; either version 2.1      *
 *  /  /  /     \   of the License, or (at your option) any later version.   *
 * (  (  ( B S C )                                                           *
 *  \  \  \_____/   This library is distributed in hope that it will be      *
 *   \  \__         useful but WITHOUT ANY WARRANTY; without even the        *
 *    \___          implied warranty of MERCHANTABILITY or FITNESS FOR A     *
 *                  PARTICULAR PURPOSE. See the GNU LGPL for more details.   *
 *                                                                           *
 * You should have received a copy of the GNU Lesser General Public License  *
 * along with this library; if not, write to the Free Software Foundation,   *
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA          *
 * The GNU LEsser General Public License is contained in the file COPYING.   *
 *                                 ---------                                 *
 *   Barcelona Supercomputing Center - Centro Nacional de Supercomputacion   *
\*****************************************************************************/


#include "event_list.h"
#include "event_description.h"
#include "MPI_EventEncoding.h"
#include "events.h"
#include "papi_events.h"

EventList *EventList::instance = nullptr;

EventList *EventList::getInstance()
{
  if( instance == nullptr )
    instance = new EventList();
  return instance;
}

EventList::EventList()
{}

EventList::~EventList()
{
  for( vector<EventDescription *>::iterator it = events.begin();
       it != events.end(); ++it )
    delete *it;
}

bool EventList::insert( string          stringID,
                        bool            usedInExtrae,
                        prvEventType_t  mpitID,
                        prvEventType_t  whichType,
                        prvEventValue_t whichValue,
                        string          whichStrType,
                        string          whichStrValue,
                        bool            isChangingState,
                        prvState_t      whichStateTransition )
{
  if( stringMap.find( stringID ) != stringMap.end() ||
      mpitMap.find( mpitID ) != mpitMap.end() ||
      ( typeValueMap.find( whichType ) != typeValueMap.end() &&
        ( ( *typeValueMap.find( whichType ) ).second ).find( whichValue ) != ( *typeValueMap.find( whichType ) ).second.end() )
    )
    return false;

  EventDescription *tmp = new EventDescription( usedInExtrae,
                                                whichType,
                                                whichValue,
                                                whichStrType,
                                                whichStrValue,
                                                isChangingState,
                                                whichStateTransition );

  events.push_back( tmp );
  if( stringID != "" )
    stringMap[ stringID ] = tmp;
  if( usedInExtrae )
    mpitMap[ mpitID ] = tmp;
  if( typeValueMap.find( whichType ) == typeValueMap.end() )
    typeValueMap[ whichType ] = map<prvEventValue_t, EventDescription *>();
  ( typeValueMap[ whichType ] )[ whichValue ] = tmp;

  return true;
}

bool EventList::insert( string            stringID,
                        prvEventType_t    mpitID,
                        EventDescription *whichEvent )
{
  prvEventType_t whichType = whichEvent->getType();
  prvEventValue_t whichValue = whichEvent->getValue();

  if( stringMap.find( stringID ) != stringMap.end() ||
      mpitMap.find( mpitID ) != mpitMap.end() ||
      ( typeValueMap.find( whichType ) != typeValueMap.end() &&
        ( ( *typeValueMap.find( whichType ) ).second ).find( whichValue ) != ( *typeValueMap.find( whichType ) ).second.end() )
    )
    return false;

  events.push_back( whichEvent );
  if( stringID != "" )
    stringMap[ stringID ] = whichEvent;
  if( whichEvent->getUsedInExtrae() )
    mpitMap[ mpitID ] = whichEvent;
  if( typeValueMap.find( whichType ) == typeValueMap.end() )
    typeValueMap[ whichType ] = map<prvEventValue_t, EventDescription *>();
  ( typeValueMap[ whichType ] )[ whichValue ] = whichEvent;

}

EventDescription *EventList::getByStringID( string whichStringID ) const
{
  map<string, EventDescription *>::const_iterator it;
  if( ( it = stringMap.find( whichStringID ) ) == stringMap.end() )
    return nullptr;
  return ( *it ).second;
}

EventDescription *EventList::getByMpitID( prvEventType_t whichMpitID ) const
{
  map<prvEventType_t, EventDescription *>::const_iterator it;
  if( ( it = mpitMap.find( whichMpitID ) ) == mpitMap.end() )
    return nullptr;
  return ( *it ).second;
}

EventDescription *EventList::getByTypeValue( prvEventType_t  whichType,
                                             prvEventValue_t whichValue ) const
{
  map<prvEventType_t, map<prvEventValue_t, EventDescription *> >::const_iterator itType;
  map<prvEventValue_t, EventDescription *>::const_iterator itValue;
  if( ( itType = typeValueMap.find( whichType ) ) == typeValueMap.end() )
    return nullptr;
  else if( ( itValue = ( *itType ).second.find( whichValue ) ) == ( *itType ).second.end() )
    return nullptr;

  return ( *itValue ).second;
}

void EventList::getUsed( vector<EventDescription *>& onVector ) const
{
  map<prvEventType_t, map<prvEventValue_t, EventDescription *> >::const_iterator itType;
  map<prvEventValue_t, EventDescription *>::const_iterator itValue;

  for( itType = typeValueMap.begin(); itType != typeValueMap.end(); ++itType )
  {
    for( itValue = ( *itType ).second.begin(); itValue != ( *itType ).second.end(); ++itValue )
    {
      if( ( *itValue ).second->getUsed() )
        onVector.push_back( ( *itValue ).second );
    }
  }
}

void EventList::init()
{
/************************************************************************
 *******                      MPI Events                          *******
 ************************************************************************/
  insert( MPI_SEND_LABEL, true, MPI_SEND_EV, MPITYPE_PTOP, MPI_SEND_VAL, MPITYPE_PTOP_LABEL, MPI_SEND_LABEL, true, STATE_SEND );
  insert( MPI_RECV_LABEL, true, MPI_RECV_EV, MPITYPE_PTOP, MPI_RECV_VAL, MPITYPE_PTOP_LABEL, MPI_RECV_LABEL, true, STATE_WAITMESS );
  insert( MPI_ISEND_LABEL, true, MPI_ISEND_EV, MPITYPE_PTOP, MPI_ISEND_VAL, MPITYPE_PTOP_LABEL, MPI_ISEND_LABEL, true, STATE_ISEND );
  insert( MPI_IRECV_LABEL, true, MPI_IRECV_EV, MPITYPE_PTOP, MPI_IRECV_VAL, MPITYPE_PTOP_LABEL, MPI_IRECV_LABEL, true, STATE_IWAITMESS );
  insert( MPI_WAIT_LABEL, true, MPI_WAIT_EV, MPITYPE_PTOP, MPI_WAIT_VAL, MPITYPE_PTOP_LABEL, MPI_WAIT_LABEL, true, STATE_TWRECV );
  insert( MPI_WAITALL_LABEL, true, MPI_WAITALL_EV, MPITYPE_PTOP, MPI_WAITALL_VAL, MPITYPE_PTOP_LABEL, MPI_WAITALL_LABEL, true, STATE_TWRECV );
  insert( MPI_BCAST_LABEL, true, MPI_BCAST_EV, MPITYPE_COLLECTIVE, MPI_BCAST_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_BCAST_LABEL, true, STATE_BCAST );
  insert( MPI_BARRIER_LABEL, true, MPI_BARRIER_EV, MPITYPE_COLLECTIVE, MPI_BARRIER_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_BARRIER_LABEL, true, STATE_BARRIER );
  insert( MPI_REDUCE_LABEL, true, MPI_REDUCE_EV, MPITYPE_COLLECTIVE, MPI_REDUCE_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_REDUCE_LABEL, true, STATE_BCAST );
  insert( MPI_ALLREDUCE_LABEL, true, MPI_ALLREDUCE_EV, MPITYPE_COLLECTIVE, MPI_ALLREDUCE_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_ALLREDUCE_LABEL, true, STATE_BCAST );
  insert( MPI_ALLTOALL_LABEL, true, MPI_ALLTOALL_EV, MPITYPE_COLLECTIVE, MPI_ALLTOALL_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_ALLTOALL_LABEL, true, STATE_BCAST );
  insert( MPI_ALLTOALLV_LABEL, true, MPI_ALLTOALLV_EV, MPITYPE_COLLECTIVE, MPI_ALLTOALLV_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_ALLTOALLV_LABEL, true, STATE_BCAST );
  insert( MPI_GATHER_LABEL, true, MPI_GATHER_EV, MPITYPE_COLLECTIVE, MPI_GATHER_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_GATHER_LABEL, true, STATE_BCAST );
  insert( MPI_GATHERV_LABEL, true, MPI_GATHERV_EV, MPITYPE_COLLECTIVE, MPI_GATHERV_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_GATHERV_LABEL, true, STATE_BCAST );
  insert( MPI_SCATTER_LABEL, true, MPI_SCATTER_EV, MPITYPE_COLLECTIVE, MPI_SCATTER_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_SCATTER_LABEL, true, STATE_BCAST );
  insert( MPI_SCATTERV_LABEL, true, MPI_SCATTERV_EV, MPITYPE_COLLECTIVE, MPI_SCATTERV_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_SCATTERV_LABEL, true, STATE_BCAST );
  insert( MPI_ALLGATHER_LABEL, true, MPI_ALLGATHER_EV, MPITYPE_COLLECTIVE, MPI_ALLGATHER_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_ALLGATHER_LABEL, true, STATE_BCAST );
  insert( MPI_ALLGATHERV_LABEL, true, MPI_ALLGATHERV_EV, MPITYPE_COLLECTIVE, MPI_ALLGATHERV_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_ALLGATHERV_LABEL, true, STATE_BCAST );
  insert( MPI_SCAN_LABEL, true, MPI_SCAN_EV, MPITYPE_COLLECTIVE, MPI_SCAN_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_SCAN_LABEL, true, STATE_BCAST );
  insert( MPI_INIT_LABEL, true, MPI_INIT_EV, MPITYPE_OTHER, MPI_INIT_VAL, MPITYPE_OTHER_LABEL, MPI_INIT_LABEL, true, STATE_INITFINI );
  insert( MPI_FINALIZE_LABEL, true, MPI_FINALIZE_EV, MPITYPE_OTHER, MPI_FINALIZE_VAL, MPITYPE_OTHER_LABEL, MPI_FINALIZE_LABEL, true, STATE_INITFINI );
  insert( MPI_BSEND_LABEL, true, MPI_BSEND_EV, MPITYPE_PTOP, MPI_BSEND_VAL, MPITYPE_PTOP_LABEL, MPI_BSEND_LABEL, true, STATE_SEND );
  insert( MPI_SSEND_LABEL, true, MPI_SSEND_EV, MPITYPE_PTOP, MPI_SSEND_VAL, MPITYPE_PTOP_LABEL, MPI_SSEND_LABEL, true, STATE_SEND );
  insert( MPI_RSEND_LABEL, true, MPI_RSEND_EV, MPITYPE_PTOP, MPI_RSEND_VAL, MPITYPE_PTOP_LABEL, MPI_RSEND_LABEL, true, STATE_SEND );
  insert( MPI_IBSEND_LABEL, true, MPI_IBSEND_EV, MPITYPE_PTOP, MPI_IBSEND_VAL, MPITYPE_PTOP_LABEL, MPI_IBSEND_LABEL, true, STATE_ISEND );
  insert( MPI_ISSEND_LABEL, true, MPI_ISSEND_EV, MPITYPE_PTOP, MPI_ISSEND_VAL, MPITYPE_PTOP_LABEL, MPI_ISSEND_LABEL, true, STATE_ISEND );
  insert( MPI_IRSEND_LABEL, true, MPI_IRSEND_EV, MPITYPE_PTOP, MPI_IRSEND_VAL, MPITYPE_PTOP_LABEL, MPI_IRSEND_LABEL, true, STATE_ISEND );
  insert( MPI_TEST_LABEL, true, MPI_TEST_EV, MPITYPE_OTHER, MPI_TEST_VAL, MPITYPE_OTHER_LABEL, MPI_TEST_LABEL, true, STATE_TWRECV );
  insert( MPI_CANCEL_LABEL, true, MPI_CANCEL_EV, MPITYPE_PTOP, MPI_CANCEL_VAL, MPITYPE_PTOP_LABEL, MPI_CANCEL_LABEL, true, STATE_MIXED );
  insert( MPI_SENDRECV_LABEL, true, MPI_SENDRECV_EV, MPITYPE_PTOP, MPI_SENDRECV_VAL, MPITYPE_PTOP_LABEL, MPI_SENDRECV_LABEL, true, STATE_SENDRECVOP );
  insert( MPI_SENDRECV_REPLACE_LABEL, true, MPI_SENDRECV_REPLACE_EV, MPITYPE_PTOP, MPI_SENDRECV_REPLACE_VAL, MPITYPE_PTOP_LABEL, MPI_SENDRECV_REPLACE_LABEL, true, STATE_SENDRECVOP );
  insert( MPI_CART_CREATE_LABEL, true, MPI_CART_CREATE_EV, MPITYPE_TOPOLOGIES, MPI_CART_CREATE_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_CART_CREATE_LABEL, false, 0 );
  insert( MPI_CART_SHIFT_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_CART_SHIFT_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_CART_SHIFT_LABEL, false, 0 );
  insert( MPI_CART_COORDS_LABEL, true, MPI_CART_COORDS_EV, MPITYPE_TOPOLOGIES, MPI_CART_COORDS_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_CART_COORDS_LABEL, false, 0 );
  insert( MPI_CART_GET_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_CART_GET_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_CART_GET_LABEL, false, 0 );
  insert( MPI_CART_MAP_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_CART_MAP_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_CART_MAP_LABEL, false, 0 );
  insert( MPI_CART_RANK_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_CART_RANK_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_CART_RANK_LABEL, false, 0 );
  insert( MPI_CART_SUB_LABEL, true, MPI_CART_SUB_EV, MPITYPE_TOPOLOGIES, MPI_CART_SUB_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_CART_SUB_LABEL, false, 0 );
  insert( MPI_CARTDIM_GET_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_CARTDIM_GET_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_CARTDIM_GET_LABEL, false, 0 );
  insert( MPI_DIMS_CREATE_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_DIMS_CREATE_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_DIMS_CREATE_LABEL, false, 0 );
  insert( MPI_GRAPH_GET_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_GRAPH_GET_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_GRAPH_GET_LABEL, false, 0 );
  insert( MPI_GRAPH_MAP_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_GRAPH_MAP_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_GRAPH_MAP_LABEL, false, 0 );
  insert( MPI_GRAPH_CREATE_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_GRAPH_CREATE_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_GRAPH_CREATE_LABEL, false, 0 );
  insert( MPI_GRAPH_NEIGHBORS_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_GRAPH_NEIGHBORS_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_GRAPH_NEIGHBORS_LABEL, false, 0 );
  insert( MPI_GRAPHDIMS_GET_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_GRAPHDIMS_GET_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_GRAPHDIMS_GET_LABEL, false, 0 );
  insert( MPI_GRAPH_NEIGHBORS_COUNT_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_GRAPH_NEIGHBORS_COUNT_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_GRAPH_NEIGHBORS_COUNT_LABEL, false, 0 );
  insert( MPI_TOPO_TEST_LABEL, false, 0, MPITYPE_TOPOLOGIES, MPI_TOPO_TEST_VAL, MPITYPE_TOPOLOGIES_LABEL, MPI_TOPO_TEST_LABEL, false, 0 );
  insert( MPI_WAITANY_LABEL, true, MPI_WAITANY_EV, MPITYPE_PTOP, MPI_WAITANY_VAL, MPITYPE_PTOP_LABEL, MPI_WAITANY_LABEL, true, STATE_TWRECV );
  insert( MPI_WAITSOME_LABEL, true, MPI_WAITSOME_EV, MPITYPE_PTOP, MPI_WAITSOME_VAL, MPITYPE_PTOP_LABEL, MPI_WAITSOME_LABEL, true, STATE_TWRECV );
  insert( MPI_PROBE_LABEL, true, MPI_PROBE_EV, MPITYPE_PTOP, MPI_PROBE_VAL, MPITYPE_PTOP_LABEL, MPI_PROBE_LABEL, true, STATE_PROBE );
  insert( MPI_IPROBE_LABEL, true, MPI_IPROBE_EV, MPITYPE_PTOP, MPI_IPROBE_VAL, MPITYPE_PTOP_LABEL, MPI_IPROBE_LABEL, true, STATE_PROBE );
  insert( MPI_WIN_CREATE_LABEL, false, 0, MPITYPE_RMA, MPI_WIN_CREATE_VAL, MPITYPE_RMA_LABEL, MPI_WIN_CREATE_LABEL, false, 0 );
  insert( MPI_WIN_FREE_LABEL, false, 0, MPITYPE_RMA, MPI_WIN_FREE_VAL, MPITYPE_RMA_LABEL, MPI_WIN_FREE_LABEL, false, 0 );
  insert( MPI_PUT_LABEL, true, MPI_PUT_EV, MPITYPE_RMA, MPI_PUT_VAL, MPITYPE_RMA_LABEL, MPI_PUT_LABEL, true, STATE_MEMORY_XFER );
  insert( MPI_GET_LABEL, true, MPI_GET_EV, MPITYPE_RMA, MPI_GET_VAL, MPITYPE_RMA_LABEL, MPI_GET_LABEL, true, STATE_MEMORY_XFER );
  insert( MPI_ACCUMULATE_LABEL, false, 0, MPITYPE_RMA, MPI_ACCUMULATE_VAL, MPITYPE_RMA_LABEL, MPI_ACCUMULATE_LABEL, false, 0 );
  insert( MPI_WIN_FENCE_LABEL, false, 0, MPITYPE_RMA, MPI_WIN_FENCE_VAL, MPITYPE_RMA_LABEL, MPI_WIN_FENCE_LABEL, false, 0 );
  insert( MPI_WIN_START_LABEL, false, 0, MPITYPE_RMA, MPI_WIN_START_VAL, MPITYPE_RMA_LABEL, MPI_WIN_START_LABEL, false, 0 );
  insert( MPI_WIN_COMPLETE_LABEL, false, 0, MPITYPE_RMA, MPI_WIN_COMPLETE_VAL, MPITYPE_RMA_LABEL, MPI_WIN_COMPLETE_LABEL, false, 0 );
  insert( MPI_WIN_POST_LABEL, false, 0, MPITYPE_RMA, MPI_WIN_POST_VAL, MPITYPE_RMA_LABEL, MPI_WIN_POST_LABEL, false, 0 );
  insert( MPI_WIN_WAIT_LABEL, false, 0, MPITYPE_RMA, MPI_WIN_WAIT_VAL, MPITYPE_RMA_LABEL, MPI_WIN_WAIT_LABEL, false, 0 );
  insert( MPI_WIN_TEST_LABEL, false, 0, MPITYPE_RMA, MPI_WIN_TEST_VAL, MPITYPE_RMA_LABEL, MPI_WIN_TEST_LABEL, false, 0 );
  insert( MPI_WIN_LOCK_LABEL, false, 0, MPITYPE_RMA, MPI_WIN_LOCK_VAL, MPITYPE_RMA_LABEL, MPI_WIN_LOCK_LABEL, false, 0 );
  insert( MPI_WIN_UNLOCK_LABEL, false, 0, MPITYPE_RMA, MPI_WIN_UNLOCK_VAL, MPITYPE_RMA_LABEL, MPI_WIN_UNLOCK_LABEL, false, 0 );
  insert( MPI_PACK_LABEL, false, 0, MPITYPE_OTHER, MPI_PACK_VAL, MPITYPE_OTHER_LABEL, MPI_PACK_LABEL, false, 0 );
  insert( MPI_UNPACK_LABEL, false, 0, MPITYPE_OTHER, MPI_UNPACK_VAL, MPITYPE_OTHER_LABEL, MPI_UNPACK_LABEL, false, 0 );
  insert( MPI_OP_CREATE_LABEL, false, 0, MPITYPE_OTHER, MPI_OP_CREATE_VAL, MPITYPE_OTHER_LABEL, MPI_OP_CREATE_LABEL, false, 0 );
  insert( MPI_OP_FREE_LABEL, false, 0, MPITYPE_OTHER, MPI_OP_FREE_VAL, MPITYPE_OTHER_LABEL, MPI_OP_FREE_LABEL, false, 0 );
  insert( MPI_REDUCE_SCATTER_LABEL, true, MPI_REDUCESCAT_EV, MPITYPE_COLLECTIVE, MPI_REDUCE_SCATTER_VAL, MPITYPE_COLLECTIVE_LABEL, MPI_REDUCE_SCATTER_LABEL, true, STATE_BCAST );
  insert( MPI_ATTR_DELETE_LABEL, false, 0, MPITYPE_OTHER, MPI_ATTR_DELETE_VAL, MPITYPE_OTHER_LABEL, MPI_ATTR_DELETE_LABEL, false, 0 );
  insert( MPI_ATTR_GET_LABEL, false, 0, MPITYPE_OTHER, MPI_ATTR_GET_VAL, MPITYPE_OTHER_LABEL, MPI_ATTR_GET_LABEL, false, 0 );
  insert( MPI_ATTR_PUT_LABEL, false, 0, MPITYPE_OTHER, MPI_ATTR_PUT_VAL, MPITYPE_OTHER_LABEL, MPI_ATTR_PUT_LABEL, false, 0 );
  insert( MPI_COMM_RANK_LABEL, true, MPI_COMM_RANK_EV, MPITYPE_COMM, MPI_COMM_RANK_VAL, MPITYPE_COMM_LABEL, MPI_COMM_RANK_LABEL, true, STATE_MIXED );
  insert( MPI_COMM_SIZE_LABEL, true, MPI_COMM_SIZE_EV, MPITYPE_COMM, MPI_COMM_SIZE_VAL, MPITYPE_COMM_LABEL, MPI_COMM_SIZE_LABEL, true, STATE_MIXED );
  insert( MPI_COMM_CREATE_LABEL, true, MPI_COMM_CREATE_EV, MPITYPE_COMM, MPI_COMM_CREATE_VAL, MPITYPE_COMM_LABEL, MPI_COMM_CREATE_LABEL, false, 0 );
  insert( MPI_COMM_DUP_LABEL, true, MPI_COMM_DUP_EV, MPITYPE_COMM, MPI_COMM_DUP_VAL, MPITYPE_COMM_LABEL, MPI_COMM_DUP_LABEL, false, 0 );
  insert( MPI_COMM_SPLIT_LABEL, true, MPI_COMM_SPLIT_EV, MPITYPE_COMM, MPI_COMM_SPLIT_VAL, MPITYPE_COMM_LABEL, MPI_COMM_SPLIT_LABEL, false, 0 );
  insert( MPI_COMM_GROUP_LABEL, false, 0, MPITYPE_COMM, MPI_COMM_GROUP_VAL, MPITYPE_COMM_LABEL, MPI_COMM_GROUP_LABEL, false, 0 );
  insert( MPI_COMM_FREE_LABEL, false, 0, MPITYPE_COMM, MPI_COMM_FREE_VAL, MPITYPE_COMM_LABEL, MPI_COMM_FREE_LABEL, false, 0 );
  insert( MPI_COMM_REMOTE_GROUP_LABEL, false, 0, MPITYPE_COMM, MPI_COMM_REMOTE_GROUP_VAL, MPITYPE_COMM_LABEL, MPI_COMM_REMOTE_GROUP_LABEL, false, 0 );
  insert( MPI_COMM_REMOTE_SIZE_LABEL, false, 0, MPITYPE_COMM, MPI_COMM_REMOTE_SIZE_VAL, MPITYPE_COMM_LABEL, MPI_COMM_REMOTE_SIZE_LABEL, false, 0 );
  insert( MPI_COMM_TEST_INTER_LABEL, false, 0, MPITYPE_COMM, MPI_COMM_TEST_INTER_VAL, MPITYPE_COMM_LABEL, MPI_COMM_TEST_INTER_LABEL, false, 0 );
  insert( MPI_COMM_COMPARE_LABEL, false, 0, MPITYPE_COMM, MPI_COMM_COMPARE_VAL, MPITYPE_COMM_LABEL, MPI_COMM_COMPARE_LABEL, false, 0 );
  insert( MPI_GROUP_DIFFERENCE_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_DIFFERENCE_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_DIFFERENCE_LABEL, false, 0 );
  insert( MPI_GROUP_EXCL_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_EXCL_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_EXCL_LABEL, false, 0 );
  insert( MPI_GROUP_FREE_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_FREE_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_FREE_LABEL, false, 0 );
  insert( MPI_GROUP_INCL_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_INCL_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_INCL_LABEL, false, 0 );
  insert( MPI_GROUP_INTERSECTION_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_INTERSECTION_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_INTERSECTION_LABEL, false, 0 );
  insert( MPI_GROUP_RANK_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_RANK_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_RANK_LABEL, false, 0 );
  insert( MPI_GROUP_RANGE_EXCL_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_RANGE_EXCL_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_RANGE_EXCL_LABEL, false, 0 );
  insert( MPI_GROUP_RANGE_INCL_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_RANGE_INCL_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_RANGE_INCL_LABEL, false, 0 );
  insert( MPI_GROUP_SIZE_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_SIZE_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_SIZE_LABEL, false, 0 );
  insert( MPI_GROUP_TRANSLATE_RANKS_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_TRANSLATE_RANKS_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_TRANSLATE_RANKS_LABEL, false, 0 );
  insert( MPI_GROUP_UNION_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_UNION_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_UNION_LABEL, false, 0 );
  insert( MPI_GROUP_COMPARE_LABEL, false, 0, MPITYPE_GROUP, MPI_GROUP_COMPARE_VAL, MPITYPE_GROUP_LABEL, MPI_GROUP_COMPARE_LABEL, false, 0 );
  insert( MPI_INTERCOMM_CREATE_LABEL, false, 0, MPITYPE_COMM, MPI_INTERCOMM_CREATE_VAL, MPITYPE_COMM_LABEL, MPI_INTERCOMM_CREATE_LABEL, false, 0 );
  insert( MPI_INTERCOMM_MERGE_LABEL, false, 0, MPITYPE_COMM, MPI_INTERCOMM_MERGE_VAL, MPITYPE_COMM_LABEL, MPI_INTERCOMM_MERGE_LABEL, false, 0 );
  insert( MPI_KEYVAL_FREE_LABEL, false, 0, MPITYPE_OTHER, MPI_KEYVAL_FREE_VAL, MPITYPE_OTHER_LABEL, MPI_KEYVAL_FREE_LABEL, false, 0 );
  insert( MPI_KEYVAL_CREATE_LABEL, false, 0, MPITYPE_OTHER, MPI_KEYVAL_CREATE_VAL, MPITYPE_OTHER_LABEL, MPI_KEYVAL_CREATE_LABEL, false, 0 );
  insert( MPI_ABORT_LABEL, false, 0, MPITYPE_OTHER, MPI_ABORT_VAL, MPITYPE_OTHER_LABEL, MPI_ABORT_LABEL, false, 0 );
  insert( MPI_ERROR_CLASS_LABEL, false, 0, MPITYPE_OTHER, MPI_ERROR_CLASS_VAL, MPITYPE_OTHER_LABEL, MPI_ERROR_CLASS_LABEL, false, 0 );
  insert( MPI_ERRHANDLER_CREATE_LABEL, false, 0, MPITYPE_OTHER, MPI_ERRHANDLER_CREATE_VAL, MPITYPE_OTHER_LABEL, MPI_ERRHANDLER_CREATE_LABEL, false, 0 );
  insert( MPI_ERRHANDLER_FREE_LABEL, false, 0, MPITYPE_OTHER, MPI_ERRHANDLER_FREE_VAL, MPITYPE_OTHER_LABEL, MPI_ERRHANDLER_FREE_LABEL, false, 0 );
  insert( MPI_ERRHANDLER_GET_LABEL, false, 0, MPITYPE_OTHER, MPI_ERRHANDLER_GET_VAL, MPITYPE_OTHER_LABEL, MPI_ERRHANDLER_GET_LABEL, false, 0 );
  insert( MPI_ERROR_STRING_LABEL, false, 0, MPITYPE_OTHER, MPI_ERROR_STRING_VAL, MPITYPE_OTHER_LABEL, MPI_ERROR_STRING_LABEL, false, 0 );
  insert( MPI_ERRHANDLER_SET_LABEL, false, 0, MPITYPE_OTHER, MPI_ERRHANDLER_SET_VAL, MPITYPE_OTHER_LABEL, MPI_ERRHANDLER_SET_LABEL, false, 0 );
  insert( MPI_GET_PROCESSOR_NAME_LABEL, false, 0, MPITYPE_OTHER, MPI_GET_PROCESSOR_NAME_VAL, MPITYPE_OTHER_LABEL, MPI_GET_PROCESSOR_NAME_LABEL, false, 0 );
  insert( MPI_INITIALIZED_LABEL, false, 0, MPITYPE_OTHER, MPI_INITIALIZED_VAL, MPITYPE_OTHER_LABEL, MPI_INITIALIZED_LABEL, false, 0 );
  insert( MPI_WTICK_LABEL, false, 0, MPITYPE_OTHER, MPI_WTICK_VAL, MPITYPE_OTHER_LABEL, MPI_WTICK_LABEL, false, 0 );
  insert( MPI_WTIME_LABEL, false, 0, MPITYPE_OTHER, MPI_WTIME_VAL, MPITYPE_OTHER_LABEL, MPI_WTIME_LABEL, false, 0 );
  insert( MPI_ADDRESS_LABEL, false, 0, MPITYPE_OTHER, MPI_ADDRESS_VAL, MPITYPE_OTHER_LABEL, MPI_ADDRESS_LABEL, false, 0 );
  insert( MPI_BSEND_INIT_LABEL, true, MPI_BSEND_INIT_EV, MPITYPE_PTOP, MPI_BSEND_INIT_VAL, MPITYPE_PTOP_LABEL, MPI_BSEND_INIT_LABEL, true, STATE_ISEND );
  insert( MPI_BUFFER_ATTACH_LABEL, false, 0, MPITYPE_OTHER, MPI_BUFFER_ATTACH_VAL, MPITYPE_OTHER_LABEL, MPI_BUFFER_ATTACH_LABEL, false, 0 );
  insert( MPI_BUFFER_DETACH_LABEL, false, 0, MPITYPE_OTHER, MPI_BUFFER_DETACH_VAL, MPITYPE_OTHER_LABEL, MPI_BUFFER_DETACH_LABEL, false, 0 );
  insert( MPI_REQUEST_FREE_LABEL, true, MPI_REQUEST_FREE_EV, MPITYPE_OTHER, MPI_REQUEST_FREE_VAL, MPITYPE_OTHER_LABEL, MPI_REQUEST_FREE_LABEL, true, STATE_TWRECV );
  insert( MPI_RECV_INIT_LABEL, true, MPI_RECV_INIT_EV, MPITYPE_PTOP, MPI_RECV_INIT_VAL, MPITYPE_PTOP_LABEL, MPI_RECV_INIT_LABEL, true, STATE_IRECV );
  insert( MPI_SEND_INIT_LABEL, true, MPI_SEND_INIT_EV, MPITYPE_PTOP, MPI_SEND_INIT_VAL, MPITYPE_PTOP_LABEL, MPI_SEND_INIT_LABEL, true, STATE_ISEND );
  insert( MPI_GET_COUNT_LABEL, false, 0, MPITYPE_OTHER, MPI_GET_COUNT_VAL, MPITYPE_OTHER_LABEL, MPI_GET_COUNT_LABEL, false, 0 );
  insert( MPI_GET_ELEMENTS_LABEL, false, 0, MPITYPE_OTHER, MPI_GET_ELEMENTS_VAL, MPITYPE_OTHER_LABEL, MPI_GET_ELEMENTS_LABEL, false, 0 );
  insert( MPI_PACK_SIZE_LABEL, false, 0, MPITYPE_OTHER, MPI_PACK_SIZE_VAL, MPITYPE_OTHER_LABEL, MPI_PACK_SIZE_LABEL, false, 0 );
  insert( MPI_RSEND_INIT_LABEL, true, MPI_RSEND_INIT_EV, MPITYPE_PTOP, MPI_RSEND_INIT_VAL, MPITYPE_PTOP_LABEL, MPI_RSEND_INIT_LABEL, true, STATE_ISEND );
  insert( MPI_SSEND_INIT_LABEL, true, MPI_SSEND_INIT_EV, MPITYPE_PTOP, MPI_SSEND_INIT_VAL, MPITYPE_PTOP_LABEL, MPI_SSEND_INIT_LABEL, true, STATE_ISEND );
  insert( MPI_START_LABEL, true, MPI_START_EV, MPITYPE_OTHER, MPI_START_VAL, MPITYPE_OTHER_LABEL, MPI_START_LABEL, true, STATE_MIXED );
  insert( MPI_STARTALL_LABEL, true, MPI_STARTALL_EV, MPITYPE_OTHER, MPI_STARTALL_VAL, MPITYPE_OTHER_LABEL, MPI_STARTALL_LABEL, true, STATE_MIXED );
  insert( MPI_TESTALL_LABEL, false, 0, MPITYPE_OTHER, MPI_TESTALL_VAL, MPITYPE_OTHER_LABEL, MPI_TESTALL_LABEL, false, 0 );
  insert( MPI_TESTANY_LABEL, false, 0, MPITYPE_OTHER, MPI_TESTANY_VAL, MPITYPE_OTHER_LABEL, MPI_TESTANY_LABEL, false, 0 );
  insert( MPI_TEST_CANCELLED_LABEL, false, 0, MPITYPE_OTHER, MPI_TEST_CANCELLED_VAL, MPITYPE_OTHER_LABEL, MPI_TEST_CANCELLED_LABEL, false, 0 );
  insert( MPI_TESTSOME_LABEL, false, 0, MPITYPE_OTHER, MPI_TESTSOME_VAL, MPITYPE_OTHER_LABEL, MPI_TESTSOME_LABEL, false, 0 );
  insert( MPI_TYPE_COMMIT_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_COMMIT_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_COMMIT_LABEL, false, 0 );
  insert( MPI_TYPE_CONTIGUOUS_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_CONTIGUOUS_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_CONTIGUOUS_LABEL, false, 0 );
  insert( MPI_TYPE_EXTENT_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_EXTENT_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_EXTENT_LABEL, false, 0 );
  insert( MPI_TYPE_FREE_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_FREE_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_FREE_LABEL, false, 0 );
  insert( MPI_TYPE_HINDEXED_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_HINDEXED_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_HINDEXED_LABEL, false, 0 );
  insert( MPI_TYPE_HVECTOR_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_HVECTOR_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_HVECTOR_LABEL, false, 0 );
  insert( MPI_TYPE_INDEXED_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_INDEXED_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_INDEXED_LABEL, false, 0 );
  insert( MPI_TYPE_LB_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_LB_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_LB_LABEL, false, 0 );
  insert( MPI_TYPE_SIZE_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_SIZE_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_SIZE_LABEL, false, 0 );
  insert( MPI_TYPE_STRUCT_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_STRUCT_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_STRUCT_LABEL, false, 0 );
  insert( MPI_TYPE_UB_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_UB_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_UB_LABEL, false, 0 );
  insert( MPI_TYPE_VECTOR_LABEL, false, 0, MPITYPE_TYPE, MPI_TYPE_VECTOR_VAL, MPITYPE_TYPE_LABEL, MPI_TYPE_VECTOR_LABEL, false, 0 );
  insert( MPI_FILE_OPEN_LABEL, true, MPI_FILE_OPEN_EV, MPITYPE_IO, MPI_FILE_OPEN_VAL, MPITYPE_IO_LABEL, MPI_FILE_OPEN_LABEL, true, STATE_IO );
  insert( MPI_FILE_CLOSE_LABEL, true, MPI_FILE_CLOSE_EV, MPITYPE_IO, MPI_FILE_CLOSE_VAL, MPITYPE_IO_LABEL, MPI_FILE_CLOSE_LABEL, true, STATE_IO );
  insert( MPI_FILE_READ_LABEL, true, MPI_FILE_READ_EV, MPITYPE_IO, MPI_FILE_READ_VAL, MPITYPE_IO_LABEL, MPI_FILE_READ_LABEL, true, STATE_IO );
  insert( MPI_FILE_READ_ALL_LABEL, true, MPI_FILE_READ_ALL_EV, MPITYPE_IO, MPI_FILE_READ_ALL_VAL, MPITYPE_IO_LABEL, MPI_FILE_READ_ALL_LABEL, true, STATE_IO );
  insert( MPI_FILE_WRITE_LABEL, true, MPI_FILE_WRITE_EV, MPITYPE_IO, MPI_FILE_WRITE_VAL, MPITYPE_IO_LABEL, MPI_FILE_WRITE_LABEL, true, STATE_IO );
  insert( MPI_FILE_WRITE_ALL_LABEL, true, MPI_FILE_WRITE_ALL_EV, MPITYPE_IO, MPI_FILE_WRITE_ALL_VAL, MPITYPE_IO_LABEL, MPI_FILE_WRITE_ALL_LABEL, true, STATE_IO );
  insert( MPI_FILE_READ_AT_LABEL, true, MPI_FILE_READ_AT_EV, MPITYPE_IO, MPI_FILE_READ_AT_VAL, MPITYPE_IO_LABEL, MPI_FILE_READ_AT_LABEL, true, STATE_IO );
  insert( MPI_FILE_READ_AT_ALL_LABEL, true, MPI_FILE_READ_AT_ALL_EV, MPITYPE_IO, MPI_FILE_READ_AT_ALL_VAL, MPITYPE_IO_LABEL, MPI_FILE_READ_AT_ALL_LABEL, true, STATE_IO );
  insert( MPI_FILE_WRITE_AT_LABEL, true, MPI_FILE_WRITE_AT_EV, MPITYPE_IO, MPI_FILE_WRITE_AT_VAL, MPITYPE_IO_LABEL, MPI_FILE_WRITE_AT_LABEL, true, STATE_IO );
  insert( MPI_FILE_WRITE_AT_ALL_LABEL, true, MPI_FILE_WRITE_AT_ALL_EV, MPITYPE_IO, MPI_FILE_WRITE_AT_ALL_VAL, MPITYPE_IO_LABEL, MPI_FILE_WRITE_AT_ALL_LABEL, true, STATE_IO );

/************************************************************************
 *******                    OpenMP Events                         *******
 ************************************************************************/
  insert( "!$omp parallel", true, PAR_EV, PAR_EV, PAR_REG_VAL, "Parallel (OMP)", "REGION (open)", true, STATE_OVHD );
  insert( "!$omp implicit barrier", true, BARRIEROMP_EV, BARRIEROMP_EV, 1, "OpenMP barrier", "Begin", true, STATE_BARRIER );

/************************************************************************
 *******                    PAPI HWC                              *******
 ************************************************************************/
#define HWC_COUNTER_TYPE(x) HWC_BASE + (x & 0x0000FFFF)

  insert( PAPI_L1_DCM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_DCM_EV ), HWC_COUNTER_TYPE( PAPI_L1_DCM_EV ), 0, PAPI_L1_DCM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_ICM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_ICM_EV ), HWC_COUNTER_TYPE( PAPI_L1_ICM_EV ), 0, PAPI_L1_ICM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_DCM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_DCM_EV ), HWC_COUNTER_TYPE( PAPI_L2_DCM_EV ), 0, PAPI_L2_DCM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_ICM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_ICM_EV ), HWC_COUNTER_TYPE( PAPI_L2_ICM_EV ), 0, PAPI_L2_ICM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_DCM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_DCM_EV ), HWC_COUNTER_TYPE( PAPI_L3_DCM_EV ), 0, PAPI_L3_DCM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_ICM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_ICM_EV ), HWC_COUNTER_TYPE( PAPI_L3_ICM_EV ), 0, PAPI_L3_ICM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_TCM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_TCM_EV ), HWC_COUNTER_TYPE( PAPI_L1_TCM_EV ), 0, PAPI_L1_TCM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_TCM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_TCM_EV ), HWC_COUNTER_TYPE( PAPI_L2_TCM_EV ), 0, PAPI_L2_TCM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_TCM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_TCM_EV ), HWC_COUNTER_TYPE( PAPI_L3_TCM_EV ), 0, PAPI_L3_TCM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_CA_SNP_LABEL, true, HWC_COUNTER_TYPE( PAPI_CA_SNP_EV ), HWC_COUNTER_TYPE( PAPI_CA_SNP_EV ), 0, PAPI_CA_SNP_TYPE_LABEL, "", false, 0 );
  insert( PAPI_CA_SHR_LABEL, true, HWC_COUNTER_TYPE( PAPI_CA_SHR_EV ), HWC_COUNTER_TYPE( PAPI_CA_SHR_EV ), 0, PAPI_CA_SHR_TYPE_LABEL, "", false, 0 );
  insert( PAPI_CA_CLN_LABEL, true, HWC_COUNTER_TYPE( PAPI_CA_CLN_EV ), HWC_COUNTER_TYPE( PAPI_CA_CLN_EV ), 0, PAPI_CA_CLN_TYPE_LABEL, "", false, 0 );
  insert( PAPI_CA_INV_LABEL, true, HWC_COUNTER_TYPE( PAPI_CA_INV_EV ), HWC_COUNTER_TYPE( PAPI_CA_INV_EV ), 0, PAPI_CA_INV_TYPE_LABEL, "", false, 0 );
  insert( PAPI_CA_ITV_LABEL, true, HWC_COUNTER_TYPE( PAPI_CA_ITV_EV ), HWC_COUNTER_TYPE( PAPI_CA_ITV_EV ), 0, PAPI_CA_ITV_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_LDM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_LDM_EV ), HWC_COUNTER_TYPE( PAPI_L3_LDM_EV ), 0, PAPI_L3_LDM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_STM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_STM_EV ), HWC_COUNTER_TYPE( PAPI_L3_STM_EV ), 0, PAPI_L3_STM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_BRU_IDL_LABEL, true, HWC_COUNTER_TYPE( PAPI_BRU_IDL_EV ), HWC_COUNTER_TYPE( PAPI_BRU_IDL_EV ), 0, PAPI_BRU_IDL_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FXU_IDL_LABEL, true, HWC_COUNTER_TYPE( PAPI_FXU_IDL_EV ), HWC_COUNTER_TYPE( PAPI_FXU_IDL_EV ), 0, PAPI_FXU_IDL_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FPU_IDL_LABEL, true, HWC_COUNTER_TYPE( PAPI_FPU_IDL_EV ), HWC_COUNTER_TYPE( PAPI_FPU_IDL_EV ), 0, PAPI_FPU_IDL_TYPE_LABEL, "", false, 0 );
  insert( PAPI_LSU_IDL_LABEL, true, HWC_COUNTER_TYPE( PAPI_LSU_IDL_EV ), HWC_COUNTER_TYPE( PAPI_LSU_IDL_EV ), 0, PAPI_LSU_IDL_TYPE_LABEL, "", false, 0 );
  insert( PAPI_TLB_DM_LABEL, true, HWC_COUNTER_TYPE( PAPI_TLB_DM_EV ), HWC_COUNTER_TYPE( PAPI_TLB_DM_EV ), 0, PAPI_TLB_DM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_TLB_IM_LABEL, true, HWC_COUNTER_TYPE( PAPI_TLB_IM_EV ), HWC_COUNTER_TYPE( PAPI_TLB_IM_EV ), 0, PAPI_TLB_IM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_TLB_TL_LABEL, true, HWC_COUNTER_TYPE( PAPI_TLB_TL_EV ), HWC_COUNTER_TYPE( PAPI_TLB_TL_EV ), 0, PAPI_TLB_TL_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_LDM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_LDM_EV ), HWC_COUNTER_TYPE( PAPI_L1_LDM_EV ), 0, PAPI_L1_LDM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_STM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_STM_EV ), HWC_COUNTER_TYPE( PAPI_L1_STM_EV ), 0, PAPI_L1_STM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_LDM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_LDM_EV ), HWC_COUNTER_TYPE( PAPI_L2_LDM_EV ), 0, PAPI_L2_LDM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_STM_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_STM_EV ), HWC_COUNTER_TYPE( PAPI_L2_STM_EV ), 0, PAPI_L2_STM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_BTAC_M_LABEL, true, HWC_COUNTER_TYPE( PAPI_BTAC_M_EV ), HWC_COUNTER_TYPE( PAPI_BTAC_M_EV ), 0, PAPI_BTAC_M_TYPE_LABEL, "", false, 0 );
  insert( PAPI_PRF_DM_LABEL, true, HWC_COUNTER_TYPE( PAPI_PRF_DM_EV ), HWC_COUNTER_TYPE( PAPI_PRF_DM_EV ), 0, PAPI_PRF_DM_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_DCH_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_DCH_EV ), HWC_COUNTER_TYPE( PAPI_L3_DCH_EV ), 0, PAPI_L3_DCH_TYPE_LABEL, "", false, 0 );
  insert( PAPI_TLB_SD_LABEL, true, HWC_COUNTER_TYPE( PAPI_TLB_SD_EV ), HWC_COUNTER_TYPE( PAPI_TLB_SD_EV ), 0, PAPI_TLB_SD_TYPE_LABEL, "", false, 0 );
  insert( PAPI_CSR_FAL_LABEL, true, HWC_COUNTER_TYPE( PAPI_CSR_FAL_EV ), HWC_COUNTER_TYPE( PAPI_CSR_FAL_EV ), 0, PAPI_CSR_FAL_TYPE_LABEL, "", false, 0 );
  insert( PAPI_CSR_SUC_LABEL, true, HWC_COUNTER_TYPE( PAPI_CSR_SUC_EV ), HWC_COUNTER_TYPE( PAPI_CSR_SUC_EV ), 0, PAPI_CSR_SUC_TYPE_LABEL, "", false, 0 );
  insert( PAPI_CSR_TOT_LABEL, true, HWC_COUNTER_TYPE( PAPI_CSR_TOT_EV ), HWC_COUNTER_TYPE( PAPI_CSR_TOT_EV ), 0, PAPI_CSR_TOT_TYPE_LABEL, "", false, 0 );
  insert( PAPI_MEM_SCY_LABEL, true, HWC_COUNTER_TYPE( PAPI_MEM_SCY_EV ), HWC_COUNTER_TYPE( PAPI_MEM_SCY_EV ), 0, PAPI_MEM_SCY_TYPE_LABEL, "", false, 0 );
  insert( PAPI_MEM_RCY_LABEL, true, HWC_COUNTER_TYPE( PAPI_MEM_RCY_EV ), HWC_COUNTER_TYPE( PAPI_MEM_RCY_EV ), 0, PAPI_MEM_RCY_TYPE_LABEL, "", false, 0 );
  insert( PAPI_MEM_WCY_LABEL, true, HWC_COUNTER_TYPE( PAPI_MEM_WCY_EV ), HWC_COUNTER_TYPE( PAPI_MEM_WCY_EV ), 0, PAPI_MEM_WCY_TYPE_LABEL, "", false, 0 );
  insert( PAPI_STL_ICY_LABEL, true, HWC_COUNTER_TYPE( PAPI_STL_ICY_EV ), HWC_COUNTER_TYPE( PAPI_STL_ICY_EV ), 0, PAPI_STL_ICY_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FUL_ICY_LABEL, true, HWC_COUNTER_TYPE( PAPI_FUL_ICY_EV ), HWC_COUNTER_TYPE( PAPI_FUL_ICY_EV ), 0, PAPI_FUL_ICY_TYPE_LABEL, "", false, 0 );
  insert( PAPI_STL_CCY_LABEL, true, HWC_COUNTER_TYPE( PAPI_STL_CCY_EV ), HWC_COUNTER_TYPE( PAPI_STL_CCY_EV ), 0, PAPI_STL_CCY_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FUL_CCY_LABEL, true, HWC_COUNTER_TYPE( PAPI_FUL_CCY_EV ), HWC_COUNTER_TYPE( PAPI_FUL_CCY_EV ), 0, PAPI_FUL_CCY_TYPE_LABEL, "", false, 0 );
  insert( PAPI_HW_INT_LABEL, true, HWC_COUNTER_TYPE( PAPI_HW_INT_EV ), HWC_COUNTER_TYPE( PAPI_HW_INT_EV ), 0, PAPI_HW_INT_TYPE_LABEL, "", false, 0 );
  insert( PAPI_BR_UCN_LABEL, true, HWC_COUNTER_TYPE( PAPI_BR_UCN_EV ), HWC_COUNTER_TYPE( PAPI_BR_UCN_EV ), 0, PAPI_BR_UCN_TYPE_LABEL, "", false, 0 );
  insert( PAPI_BR_CN_LABEL, true, HWC_COUNTER_TYPE( PAPI_BR_CN_EV ), HWC_COUNTER_TYPE( PAPI_BR_CN_EV ), 0, PAPI_BR_CN_TYPE_LABEL, "", false, 0 );
  insert( PAPI_BR_TKN_LABEL, true, HWC_COUNTER_TYPE( PAPI_BR_TKN_EV ), HWC_COUNTER_TYPE( PAPI_BR_TKN_EV ), 0, PAPI_BR_TKN_TYPE_LABEL, "", false, 0 );
  insert( PAPI_BR_NTK_LABEL, true, HWC_COUNTER_TYPE( PAPI_BR_NTK_EV ), HWC_COUNTER_TYPE( PAPI_BR_NTK_EV ), 0, PAPI_BR_NTK_TYPE_LABEL, "", false, 0 );
  insert( PAPI_BR_MSP_LABEL, true, HWC_COUNTER_TYPE( PAPI_BR_MSP_EV ), HWC_COUNTER_TYPE( PAPI_BR_MSP_EV ), 0, PAPI_BR_MSP_TYPE_LABEL, "", false, 0 );
  insert( PAPI_BR_PRC_LABEL, true, HWC_COUNTER_TYPE( PAPI_BR_PRC_EV ), HWC_COUNTER_TYPE( PAPI_BR_PRC_EV ), 0, PAPI_BR_PRC_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FMA_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_FMA_INS_EV ), HWC_COUNTER_TYPE( PAPI_FMA_INS_EV ), 0, PAPI_FMA_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_TOT_IIS_LABEL, true, HWC_COUNTER_TYPE( PAPI_TOT_IIS_EV ), HWC_COUNTER_TYPE( PAPI_TOT_IIS_EV ), 0, PAPI_TOT_IIS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_TOT_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_TOT_INS_EV ), HWC_COUNTER_TYPE( PAPI_TOT_INS_EV ), 0, PAPI_TOT_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_INT_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_INT_INS_EV ), HWC_COUNTER_TYPE( PAPI_INT_INS_EV ), 0, PAPI_INT_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FP_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_FP_INS_EV ), HWC_COUNTER_TYPE( PAPI_FP_INS_EV ), 0, PAPI_FP_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_LD_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_LD_INS_EV ), HWC_COUNTER_TYPE( PAPI_LD_INS_EV ), 0, PAPI_LD_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_SR_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_SR_INS_EV ), HWC_COUNTER_TYPE( PAPI_SR_INS_EV ), 0, PAPI_SR_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_BR_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_BR_INS_EV ), HWC_COUNTER_TYPE( PAPI_BR_INS_EV ), 0, PAPI_BR_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_VEC_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_VEC_INS_EV ), HWC_COUNTER_TYPE( PAPI_VEC_INS_EV ), 0, PAPI_VEC_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_RES_STL_LABEL, true, HWC_COUNTER_TYPE( PAPI_RES_STL_EV ), HWC_COUNTER_TYPE( PAPI_RES_STL_EV ), 0, PAPI_RES_STL_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FP_STAL_LABEL, true, HWC_COUNTER_TYPE( PAPI_FP_STAL_EV ), HWC_COUNTER_TYPE( PAPI_FP_STAL_EV ), 0, PAPI_FP_STAL_TYPE_LABEL, "", false, 0 );
  insert( PAPI_TOT_CYC_LABEL, true, HWC_COUNTER_TYPE( PAPI_TOT_CYC_EV ), HWC_COUNTER_TYPE( PAPI_TOT_CYC_EV ), 0, PAPI_TOT_CYC_TYPE_LABEL, "", false, 0 );
  insert( PAPI_LST_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_LST_INS_EV ), HWC_COUNTER_TYPE( PAPI_LST_INS_EV ), 0, PAPI_LST_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_SYC_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_SYC_INS_EV ), HWC_COUNTER_TYPE( PAPI_SYC_INS_EV ), 0, PAPI_SYC_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_DCH_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_DCH_EV ), HWC_COUNTER_TYPE( PAPI_L1_DCH_EV ), 0, PAPI_L1_DCH_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_DCH_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_DCH_EV ), HWC_COUNTER_TYPE( PAPI_L2_DCH_EV ), 0, PAPI_L2_DCH_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_DCA_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_DCA_EV ), HWC_COUNTER_TYPE( PAPI_L1_DCA_EV ), 0, PAPI_L1_DCA_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_DCA_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_DCA_EV ), HWC_COUNTER_TYPE( PAPI_L2_DCA_EV ), 0, PAPI_L2_DCA_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_DCA_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_DCA_EV ), HWC_COUNTER_TYPE( PAPI_L3_DCA_EV ), 0, PAPI_L3_DCA_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_DCR_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_DCR_EV ), HWC_COUNTER_TYPE( PAPI_L1_DCR_EV ), 0, PAPI_L1_DCR_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_DCR_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_DCR_EV ), HWC_COUNTER_TYPE( PAPI_L2_DCR_EV ), 0, PAPI_L2_DCR_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_DCR_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_DCR_EV ), HWC_COUNTER_TYPE( PAPI_L3_DCR_EV ), 0, PAPI_L3_DCR_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_DCW_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_DCW_EV ), HWC_COUNTER_TYPE( PAPI_L1_DCW_EV ), 0, PAPI_L1_DCW_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_DCW_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_DCW_EV ), HWC_COUNTER_TYPE( PAPI_L2_DCW_EV ), 0, PAPI_L2_DCW_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_DCW_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_DCW_EV ), HWC_COUNTER_TYPE( PAPI_L3_DCW_EV ), 0, PAPI_L3_DCW_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_ICH_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_ICH_EV ), HWC_COUNTER_TYPE( PAPI_L1_ICH_EV ), 0, PAPI_L1_ICH_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_ICH_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_ICH_EV ), HWC_COUNTER_TYPE( PAPI_L2_ICH_EV ), 0, PAPI_L2_ICH_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_ICH_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_ICH_EV ), HWC_COUNTER_TYPE( PAPI_L3_ICH_EV ), 0, PAPI_L3_ICH_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_ICA_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_ICA_EV ), HWC_COUNTER_TYPE( PAPI_L1_ICA_EV ), 0, PAPI_L1_ICA_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_ICA_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_ICA_EV ), HWC_COUNTER_TYPE( PAPI_L2_ICA_EV ), 0, PAPI_L2_ICA_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_ICA_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_ICA_EV ), HWC_COUNTER_TYPE( PAPI_L3_ICA_EV ), 0, PAPI_L3_ICA_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_ICR_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_ICR_EV ), HWC_COUNTER_TYPE( PAPI_L1_ICR_EV ), 0, PAPI_L1_ICR_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_ICR_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_ICR_EV ), HWC_COUNTER_TYPE( PAPI_L2_ICR_EV ), 0, PAPI_L2_ICR_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_ICR_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_ICR_EV ), HWC_COUNTER_TYPE( PAPI_L3_ICR_EV ), 0, PAPI_L3_ICR_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_ICW_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_ICW_EV ), HWC_COUNTER_TYPE( PAPI_L1_ICW_EV ), 0, PAPI_L1_ICW_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_ICW_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_ICW_EV ), HWC_COUNTER_TYPE( PAPI_L2_ICW_EV ), 0, PAPI_L2_ICW_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_ICW_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_ICW_EV ), HWC_COUNTER_TYPE( PAPI_L3_ICW_EV ), 0, PAPI_L3_ICW_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_TCH_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_TCH_EV ), HWC_COUNTER_TYPE( PAPI_L1_TCH_EV ), 0, PAPI_L1_TCH_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_TCH_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_TCH_EV ), HWC_COUNTER_TYPE( PAPI_L2_TCH_EV ), 0, PAPI_L2_TCH_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_TCH_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_TCH_EV ), HWC_COUNTER_TYPE( PAPI_L3_TCH_EV ), 0, PAPI_L3_TCH_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_TCA_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_TCA_EV ), HWC_COUNTER_TYPE( PAPI_L1_TCA_EV ), 0, PAPI_L1_TCA_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_TCA_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_TCA_EV ), HWC_COUNTER_TYPE( PAPI_L2_TCA_EV ), 0, PAPI_L2_TCA_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_TCA_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_TCA_EV ), HWC_COUNTER_TYPE( PAPI_L3_TCA_EV ), 0, PAPI_L3_TCA_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_TCR_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_TCR_EV ), HWC_COUNTER_TYPE( PAPI_L1_TCR_EV ), 0, PAPI_L1_TCR_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_TCR_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_TCR_EV ), HWC_COUNTER_TYPE( PAPI_L2_TCR_EV ), 0, PAPI_L2_TCR_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_TCR_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_TCR_EV ), HWC_COUNTER_TYPE( PAPI_L3_TCR_EV ), 0, PAPI_L3_TCR_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L1_TCW_LABEL, true, HWC_COUNTER_TYPE( PAPI_L1_TCW_EV ), HWC_COUNTER_TYPE( PAPI_L1_TCW_EV ), 0, PAPI_L1_TCW_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L2_TCW_LABEL, true, HWC_COUNTER_TYPE( PAPI_L2_TCW_EV ), HWC_COUNTER_TYPE( PAPI_L2_TCW_EV ), 0, PAPI_L2_TCW_TYPE_LABEL, "", false, 0 );
  insert( PAPI_L3_TCW_LABEL, true, HWC_COUNTER_TYPE( PAPI_L3_TCW_EV ), HWC_COUNTER_TYPE( PAPI_L3_TCW_EV ), 0, PAPI_L3_TCW_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FML_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_FML_INS_EV ), HWC_COUNTER_TYPE( PAPI_FML_INS_EV ), 0, PAPI_FML_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FAD_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_FAD_INS_EV ), HWC_COUNTER_TYPE( PAPI_FAD_INS_EV ), 0, PAPI_FAD_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FDV_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_FDV_INS_EV ), HWC_COUNTER_TYPE( PAPI_FDV_INS_EV ), 0, PAPI_FDV_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FSQ_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_FSQ_INS_EV ), HWC_COUNTER_TYPE( PAPI_FSQ_INS_EV ), 0, PAPI_FSQ_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FNV_INS_LABEL, true, HWC_COUNTER_TYPE( PAPI_FNV_INS_EV ), HWC_COUNTER_TYPE( PAPI_FNV_INS_EV ), 0, PAPI_FNV_INS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_FP_OPS_LABEL, true, HWC_COUNTER_TYPE( PAPI_FP_OPS_EV ), HWC_COUNTER_TYPE( PAPI_FP_OPS_EV ), 0, PAPI_FP_OPS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_SP_OPS_LABEL, true, HWC_COUNTER_TYPE( PAPI_SP_OPS_EV ), HWC_COUNTER_TYPE( PAPI_SP_OPS_EV ), 0, PAPI_SP_OPS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_DP_OPS_LABEL, true, HWC_COUNTER_TYPE( PAPI_DP_OPS_EV ), HWC_COUNTER_TYPE( PAPI_DP_OPS_EV ), 0, PAPI_DP_OPS_TYPE_LABEL, "", false, 0 );
  insert( PAPI_VEC_SP_LABEL, true, HWC_COUNTER_TYPE( PAPI_VEC_SP_EV ), HWC_COUNTER_TYPE( PAPI_VEC_SP_EV ), 0, PAPI_VEC_SP_TYPE_LABEL, "", false, 0 );
  insert( PAPI_VEC_DP_LABEL, true, HWC_COUNTER_TYPE( PAPI_VEC_DP_EV ), HWC_COUNTER_TYPE( PAPI_VEC_DP_EV ), 0, PAPI_VEC_DP_TYPE_LABEL, "", false, 0 );

/*
    bool insert( string          stringID,
                 bool            usedInExtrae,
                 prvEventType_t  mpitID,
                 prvEventType_t  whichType,
                 prvEventValue_t whichValue,
                 string          whichStrType,
                 string          whichStrValue,
                 bool            isChangingState,
                 prvState_t      whichStateTransition  );
*/
}
